---
id: 1
group: "plugin-system"
dependencies: []
status: "completed"
created: "2025-11-19"
skills:
  - drupal-plugin-system
  - php
---
# Create Resource Template Plugin System Infrastructure

## Objective

Implement the foundational plugin architecture for MCP resource templates, including the plugin interface, base class, manager, and PHP attribute. This establishes the plugin discovery and instantiation system that will allow different types of MCP resources to be registered.

## Skills Required

- **drupal-plugin-system**: Deep understanding of Drupal's plugin API, DefaultPluginManager, plugin discovery via attributes
- **php**: PHP 8.3+ features including attributes, readonly properties, and strict typing

## Acceptance Criteria

- [ ] ResourceTemplateInterface defines all required methods for resource plugins
- [ ] ResourceTemplateBase provides abstract base implementation with helper methods
- [ ] ResourceTemplateManager extends DefaultPluginManager and discovers plugins via attributes
- [ ] ResourceTemplate PHP attribute is defined with proper structure
- [ ] Plugin system follows Drupal coding standards (PHPCS passes)
- [ ] PHPStan analysis passes at level 1
- [ ] All classes have comprehensive PHPDoc documentation

## Technical Requirements

Create four interconnected files following Drupal's plugin system patterns:

1. **Plugin Interface** (`src/Plugin/ResourceTemplateInterface.php`):
   - Extends `PluginInspectionInterface`
   - Methods: `getResourceType()`, `getTitle()`, `getDescription()`, `getDependencies()`, `getUriTemplate()`, `getResources()`, `getResourceContent(string $uri)`, `checkAccess(string $uri, AccountInterface $account)`
   - Return types must use proper type hints (TranslatableMarkup, array, AccessResultInterface)

2. **Plugin Base** (`src/Plugin/ResourceTemplateBase.php`):
   - Abstract class implementing ResourceTemplateInterface
   - Uses ContainerFactoryPluginInterface pattern for dependency injection
   - Helper method for URI parsing (extract entity type and ID from `drupal://entity/{type}/{id}`)
   - Default `checkAccess()` implementation returning `AccessResult::forbidden()`
   - Protected properties for common services (entity_type.manager, current_user)

3. **Plugin Manager** (`src/Plugin/ResourceTemplateManager.php`):
   - Extends `DefaultPluginManager`
   - Constructor accepts module handler, cache backend, and module list
   - Uses AttributeClassDiscovery for plugin discovery
   - Searches in `Plugin/ResourceTemplate` subdirectories
   - Implements caching with 'mcp_server:resource_templates' cache tag
   - Method to get all available resource template definitions

4. **Plugin Attribute** (`src/Attribute/ResourceTemplate.php`):
   - PHP 8 attribute with `#[Attribute(Attribute::TARGET_CLASS)]`
   - Properties: `id` (string), `label` (TranslatableMarkup), `description` (TranslatableMarkup), `dependencies` (array)
   - All properties readonly and passed via constructor

## Input Dependencies

None - this is the foundational task.

## Output Artifacts

- `src/Plugin/ResourceTemplateInterface.php`
- `src/Plugin/ResourceTemplateBase.php`
- `src/Plugin/ResourceTemplateManager.php`
- `src/Attribute/ResourceTemplate.php`
- Service definition for plugin manager in `mcp_server.services.yml`

<details>
<summary>Implementation Notes</summary>

### Plugin Interface Structure

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

interface ResourceTemplateInterface extends PluginInspectionInterface {

  /**
   * Returns the resource template type identifier.
   */
  public function getResourceType(): string;

  /**
   * Returns the human-readable title.
   */
  public function getTitle(): TranslatableMarkup;

  /**
   * Returns the description.
   */
  public function getDescription(): ?TranslatableMarkup;

  /**
   * Returns array of module machine names this resource depends on.
   */
  public function getDependencies(): array;

  /**
   * Returns the URI template pattern (e.g., 'drupal://entity/{entity_type}/{entity_id}').
   */
  public function getUriTemplate(): string;

  /**
   * Returns all available resources provided by this template.
   *
   * @return array
   *   Array of resource definitions with keys:
   *   - uri: string - The resource URI
   *   - name: string - Human-readable name
   *   - description: string - Resource description
   *   - mimeType: string - MIME type of resource content
   */
  public function getResources(): array;

  /**
   * Gets the content for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return array|null
   *   Resource content array, or NULL if not found.
   */
  public function getResourceContent(string $uri): ?array;

  /**
   * Checks access for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface;

}
```

### Plugin Base Implementation Pattern

Use constructor property promotion with dependency injection:

```php
public function __construct(
  array $configuration,
  string $plugin_id,
  mixed $plugin_definition,
  protected readonly EntityTypeManagerInterface $entityTypeManager,
  protected readonly AccountProxyInterface $currentUser,
) {
  parent::__construct($configuration, $plugin_id, $plugin_definition);
}
```

Implement static `create()` for ContainerFactoryPluginInterface:

```php
public static function create(
  ContainerInterface $container,
  array $configuration,
  $plugin_id,
  $plugin_definition,
): static {
  return new static(
    $configuration,
    $plugin_id,
    $plugin_definition,
    $container->get('entity_type.manager'),
    $container->get('current_user'),
  );
}
```

Helper method for URI parsing:

```php
/**
 * Parses a resource URI into components.
 *
 * @param string $uri
 *   URI in format 'drupal://entity/{entity_type}/{entity_id}'.
 *
 * @return array{entity_type: string, entity_id: string}|null
 *   Parsed components or NULL if invalid.
 */
protected function parseUri(string $uri): ?array {
  // Pattern: drupal://entity/{entity_type}/{entity_id}
  if (!preg_match('#^drupal://entity/([^/]+)/(.+)$#', $uri, $matches)) {
    return NULL;
  }

  return [
    'entity_type' => $matches[1],
    'entity_id' => $matches[2],
  ];
}
```

### Plugin Manager Service Definition

Add to `mcp_server.services.yml`:

```yaml
  plugin.manager.mcp_resource_template:
    class: Drupal\mcp_server\Plugin\ResourceTemplateManager
    parent: default_plugin_manager
    arguments:
      - '@module_handler'
      - '@cache.discovery'
      - '@extension.list.module'
```

### Attribute Example

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Attribute;

use Drupal\Component\Plugin\Attribute\Plugin;
use Drupal\Core\StringTranslation\TranslatableMarkup;

#[\Attribute(\Attribute::TARGET_CLASS)]
final class ResourceTemplate extends Plugin {

  /**
   * Constructs a ResourceTemplate attribute.
   *
   * @param string $id
   *   The plugin ID.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   The human-readable label.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $description
   *   The description.
   * @param array $dependencies
   *   Array of module machine names this resource depends on.
   */
  public function __construct(
    public readonly string $id,
    public readonly TranslatableMarkup $label,
    public readonly TranslatableMarkup $description,
    public readonly array $dependencies = [],
  ) {}

}
```

### Testing the Plugin System

After creating these files, verify:

```bash
# Check code standards
vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Plugin/
vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Attribute/

# Run static analysis
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Plugin/
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Attribute/

# Clear cache to discover service
vendor/bin/drush cache:rebuild
```

### Key Implementation Points

1. **Use readonly properties** wherever possible (PHP 8.1+ feature)
2. **Strict typing**: Always use `declare(strict_types=1);`
3. **No getters/setters**: Per project standards, use public readonly for access
4. **Final classes**: Plugin manager and attribute should be final
5. **Guard clauses**: Use early returns in validation methods
6. **Array operations**: Prefer functional array methods over foreach loops
7. **Documentation**: PHPDoc should explain WHY, not WHAT

</details>

## Implementation Notes

Follow existing plugin patterns in the codebase, particularly `PromptArgumentCompletionProviderManager` for reference. Ensure all type hints are explicit and leverage PHP 8.3 features like constructor property promotion and readonly properties.
