---
id: 2
group: "plugin-system"
dependencies: [1]
status: "completed"
created: "2025-11-19"
skills:
  - drupal-entity-api
  - jsonapi
---
# Implement Content Entity Resource Template Plugin

## Objective

Create the ContentEntityResourceTemplate plugin that discovers content entities with canonical URLs and provides access to them via the `drupal://entity/{entity_type}/{entity_id}` URI scheme. This plugin uses JSON:API for serialization and enforces Drupal's entity access control system.

## Skills Required

- **drupal-entity-api**: Entity type discovery, entity loading, access control system, link templates
- **jsonapi**: JSON:API normalization, serializer service, understanding of JSON:API response structure

## Acceptance Criteria

- [ ] Plugin discovers all content entity types with canonical link templates
- [ ] URI pattern `drupal://entity/{entity_type}/{entity_id}` is correctly parsed and validated
- [ ] Entity loading works for all discovered entity types
- [ ] Entity access control is enforced via `$entity->access('view', $account, TRUE)`
- [ ] JSON:API serialization produces valid JSON:API formatted output
- [ ] Plugin handles non-existent entities gracefully (returns NULL content, forbidden access)
- [ ] Plugin handles JSON:API module not being enabled (validates dependency)
- [ ] Code passes PHPCS and PHPStan checks

## Technical Requirements

Create `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php`:

1. **Plugin Annotation**:
   ```php
   #[ResourceTemplate(
     id: 'content_entity',
     label: new TranslatableMarkup('Content Entity Resources'),
     description: new TranslatableMarkup('Provides access to content entities with canonical URLs'),
     dependencies: ['jsonapi'],
   )]
   ```

2. **Entity Type Discovery**:
   - Query all entity type definitions from EntityTypeManager
   - Filter for content entities: `$definition->isSubclassOf(ContentEntityInterface::class)`
   - Filter for entities with canonical link template: `$definition->hasLinkTemplate('canonical')`
   - Expected matches: node, media, taxonomy_term, user, block_content, etc.

3. **URI Generation**:
   - For each discovered entity type, generate URI template
   - Pattern: `drupal://entity/{$entity_type_id}/{{{$entity_type->getKey('id')}}}`
   - Example: `drupal://entity/node/{nid}`, `drupal://entity/media/{mid}`

4. **Resource Metadata**:
   - `uri`: The URI template with parameter placeholder
   - `name`: Entity type label + " - {parameter}"
   - `description`: Entity type description
   - `mimeType`: `application/vnd.api+json` (JSON:API standard)

5. **Entity Loading**:
   - Parse URI to extract entity_type and entity_id
   - Validate entity type exists and has canonical link template
   - Load entity: `$this->entityTypeManager->getStorage($entity_type)->load($entity_id)`
   - Return NULL if entity doesn't exist

6. **Access Control**:
   - If entity doesn't exist: return `AccessResult::forbidden()`
   - Check entity view access: `$entity->access('view', $account, TRUE)`
   - Return the AccessResult object (preserves cache metadata)

7. **JSON:API Serialization**:
   - Inject `jsonapi.serializer` service
   - Normalize entity: `$this->serializer->normalize($entity, 'api_json')`
   - Handle serialization exceptions and return NULL on failure
   - Return normalized array structure

## Input Dependencies

- Task 1 outputs: ResourceTemplateInterface, ResourceTemplateBase, ResourceTemplate attribute
- JSON:API module (Drupal core module, must be enabled)

## Output Artifacts

- `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php`
- Working plugin discoverable by ResourceTemplateManager
- JSON:API-serialized entity data accessible via MCP resources

<details>
<summary>Implementation Notes</summary>

### Plugin Class Structure

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\ResourceTemplate;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\Attribute\ResourceTemplate;
use Drupal\mcp_server\Plugin\ResourceTemplateBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\SerializerInterface;

#[ResourceTemplate(
  id: 'content_entity',
  label: new TranslatableMarkup('Content Entity Resources'),
  description: new TranslatableMarkup('Provides access to content entities with canonical URLs'),
  dependencies: ['jsonapi'],
)]
final class ContentEntityResourceTemplate extends ResourceTemplateBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    AccountProxyInterface $currentUser,
    private readonly SerializerInterface $serializer,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $currentUser);
  }

  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('jsonapi.serializer'),
    );
  }

  // Implement interface methods...
}
```

### Entity Type Discovery Implementation

```php
public function getResources(): array {
  $resources = [];

  // Get all entity type definitions.
  $entity_types = $this->entityTypeManager->getDefinitions();

  foreach ($entity_types as $entity_type_id => $entity_type) {
    // Filter: must be content entity with canonical link template.
    if (!$entity_type->isSubclassOf(ContentEntityInterface::class)) {
      continue;
    }

    if (!$entity_type->hasLinkTemplate('canonical')) {
      continue;
    }

    // Generate resource metadata.
    $id_key = $entity_type->getKey('id');
    $resources[] = [
      'uri' => "drupal://entity/{$entity_type_id}/{{$id_key}}",
      'name' => $entity_type->getLabel() . ' - {' . $id_key . '}',
      'description' => $entity_type->getDescription() ?? '',
      'mimeType' => 'application/vnd.api+json',
    ];
  }

  return $resources;
}
```

### Entity Loading with Access Control

```php
public function getResourceContent(string $uri): ?array {
  $parsed = $this->parseUri($uri);
  if ($parsed === NULL) {
    return NULL;
  }

  $entity_type = $parsed['entity_type'];
  $entity_id = $parsed['entity_id'];

  // Validate entity type exists and has canonical.
  try {
    $definition = $this->entityTypeManager->getDefinition($entity_type);
  } catch (\Exception $e) {
    return NULL;
  }

  if (!$definition->hasLinkTemplate('canonical')) {
    return NULL;
  }

  // Load entity.
  try {
    $storage = $this->entityTypeManager->getStorage($entity_type);
    $entity = $storage->load($entity_id);
  } catch (\Exception $e) {
    return NULL;
  }

  if ($entity === NULL) {
    return NULL;
  }

  // Check access before serializing.
  if (!$entity->access('view', $this->currentUser)) {
    return NULL;
  }

  // Serialize via JSON:API.
  try {
    $normalized = $this->serializer->normalize($entity, 'api_json');
    return is_array($normalized) ? $normalized : NULL;
  } catch (\Exception $e) {
    return NULL;
  }
}
```

### Access Checking Implementation

```php
public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface {
  $parsed = $this->parseUri($uri);
  if ($parsed === NULL) {
    return AccessResult::forbidden('Invalid URI format');
  }

  $entity_type = $parsed['entity_type'];
  $entity_id = $parsed['entity_id'];

  // Validate entity type.
  try {
    $definition = $this->entityTypeManager->getDefinition($entity_type);
  } catch (\Exception $e) {
    return AccessResult::forbidden('Unknown entity type');
  }

  if (!$definition->hasLinkTemplate('canonical')) {
    return AccessResult::forbidden('Entity type does not support canonical URLs');
  }

  // Load entity.
  try {
    $storage = $this->entityTypeManager->getStorage($entity_type);
    $entity = $storage->load($entity_id);
  } catch (\Exception $e) {
    return AccessResult::forbidden('Failed to load entity');
  }

  if ($entity === NULL) {
    return AccessResult::forbidden('Entity not found');
  }

  // Delegate to entity access system.
  return $entity->access('view', $account, TRUE);
}
```

### Required Interface Implementations

```php
public function getResourceType(): string {
  return $this->getPluginId();
}

public function getTitle(): TranslatableMarkup {
  return $this->pluginDefinition['label'];
}

public function getDescription(): ?TranslatableMarkup {
  return $this->pluginDefinition['description'] ?? NULL;
}

public function getDependencies(): array {
  return $this->pluginDefinition['dependencies'] ?? [];
}

public function getUriTemplate(): string {
  return 'drupal://entity/{entity_type}/{entity_id}';
}
```

### Testing Commands

```bash
# Code standards
vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Plugin/ResourceTemplate/

# Static analysis
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Plugin/ResourceTemplate/

# Clear cache and verify plugin is discovered
vendor/bin/drush cache:rebuild
vendor/bin/drush php-eval "print_r(\Drupal::service('plugin.manager.mcp_resource_template')->getDefinitions());"
```

### Key Implementation Points

1. **Error Handling**: Use try-catch for entity type/storage operations that might throw exceptions
2. **Type Safety**: Validate return types from serializer (might not always be array)
3. **Cache Metadata**: Return AccessResult objects (not booleans) to preserve cache tags/contexts
4. **Guard Clauses**: Use early returns for validation failures
5. **Entity Types to Test**: node, media, taxonomy_term, user (common ones with canonical URLs)
6. **JSON:API Format**: Result should have 'data', 'attributes', 'relationships', 'links' keys

</details>

## Implementation Notes

Reference the JSON:API module's serializer service documentation. The JSON:API normalization process respects field-level access, so sensitive fields are automatically filtered. Test with both published and unpublished content entities to verify access control.
