---
id: 3
group: "configuration"
dependencies: [1]
status: "completed"
created: "2025-11-19"
skills:
  - drupal-config-entities
---
# Create McpResourceConfig Configuration Entity

## Objective

Implement the McpResourceConfig configuration entity that allows administrators to enable/disable resource templates and track their module dependencies. This entity stores resource configurations similarly to how McpToolConfig and McpPromptConfig work.

## Skills Required

- **drupal-config-entities**: ConfigEntityBase, entity annotations/attributes, configuration schema, entity handlers

## Acceptance Criteria

- [ ] McpResourceConfig entity class is created with all required properties
- [ ] Entity uses PHP attributes for configuration (ConfigEntityType)
- [ ] Configuration schema is defined in mcp_server.schema.yml
- [ ] Entity has methods: getResourceTemplateId(), getDescription(), getDependencies()
- [ ] Entity follows ConfigEntityBase pattern with proper config_export
- [ ] Code passes PHPCS and PHPStan checks
- [ ] Configuration can be exported/imported via drush cex/cim

## Technical Requirements

Create two files:

1. **Entity Class** (`src/Entity/McpResourceConfig.php`):
   - Extends `ConfigEntityBase`
   - Uses `ConfigEntityType` attribute
   - Properties: `id`, `label`, `resource_template_id`, `description`, `dependencies`, `status`
   - Implements getter methods for all properties
   - config_prefix: `resource`
   - admin_permission: `administer mcp server`

2. **Configuration Schema** (add to `config/schema/mcp_server.schema.yml`):
   - Define schema for `mcp_server.resource.*` configuration objects
   - Include all exported properties with proper types
   - Match structure from McpToolConfig pattern

## Input Dependencies

- Task 1 outputs: ResourceTemplateInterface (for plugin ID reference)
- Existing McpToolConfig and McpPromptConfig entities (for pattern reference)

## Output Artifacts

- `src/Entity/McpResourceConfig.php`
- Configuration schema in `config/schema/mcp_server.schema.yml`
- Entity definition available for CRUD operations

<details>
<summary>Implementation Notes</summary>

### Entity Class Structure

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the MCP Resource Configuration entity.
 *
 * Configuration entity for managing MCP resource template instances.
 */
#[ConfigEntityType(
  id: 'mcp_resource_config',
  label: new TranslatableMarkup('MCP Resource Configuration'),
  label_collection: new TranslatableMarkup('MCP Resource Configurations'),
  label_singular: new TranslatableMarkup('MCP resource configuration'),
  label_plural: new TranslatableMarkup('MCP resource configurations'),
  label_count: [
    'singular' => '@count MCP resource configuration',
    'plural' => '@count MCP resource configurations',
  ],
  handlers: [
    'list_builder' => 'Drupal\mcp_server\McpResourceConfigListBuilder',
    'form' => [
      'add' => 'Drupal\mcp_server\Form\McpResourceConfigForm',
      'edit' => 'Drupal\mcp_server\Form\McpResourceConfigForm',
      'delete' => 'Drupal\Core\Entity\EntityDeleteForm',
    ],
  ],
  config_prefix: 'resource',
  admin_permission: 'administer mcp server',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'status' => 'status',
  ],
  config_export: [
    'id',
    'label',
    'resource_template_id',
    'description',
    'dependencies',
  ],
  links: [
    'add-form' => '/admin/config/services/mcp-server/resources/add',
    'edit-form' => '/admin/config/services/mcp-server/resources/{mcp_resource_config}/edit',
    'delete-form' => '/admin/config/services/mcp-server/resources/{mcp_resource_config}/delete',
    'collection' => '/admin/config/services/mcp-server/resources',
  ],
)]
final class McpResourceConfig extends ConfigEntityBase {

  /**
   * The machine name (ID) of this resource configuration.
   */
  public readonly string $id;

  /**
   * The human-readable label.
   */
  public readonly string $label;

  /**
   * The resource template plugin ID.
   */
  public readonly string $resource_template_id;

  /**
   * Optional description from the plugin.
   */
  public readonly ?string $description;

  /**
   * Module dependencies required by this resource template.
   *
   * @var array<int, string>
   */
  public readonly array $dependencies;

  /**
   * Gets the resource template plugin ID.
   */
  public function getResourceTemplateId(): string {
    return $this->resource_template_id;
  }

  /**
   * Gets the description.
   */
  public function getDescription(): ?string {
    return $this->description;
  }

  /**
   * Gets the module dependencies.
   *
   * @return array<int, string>
   *   Array of module machine names.
   */
  public function getDependencies(): array {
    return $this->dependencies ?? [];
  }

}
```

### Configuration Schema

Add to `config/schema/mcp_server.schema.yml`:

```yaml
mcp_server.resource.*:
  type: config_entity
  label: 'MCP Resource Configuration'
  mapping:
    id:
      type: string
      label: 'Machine name'
    label:
      type: label
      label: 'Label'
    resource_template_id:
      type: string
      label: 'Resource Template Plugin ID'
    description:
      type: text
      label: 'Description'
    dependencies:
      type: sequence
      label: 'Module Dependencies'
      sequence:
        type: string
        label: 'Module machine name'
    status:
      type: boolean
      label: 'Enabled'
```

### Entity Handler Placeholders

Note: The handlers referenced in the attribute (`McpResourceConfigListBuilder`, `McpResourceConfigForm`) will be created in later tasks. For now, we're defining the entity structure.

If you want to test the entity before those handlers exist, you can temporarily comment out the `handlers` array in the attribute, or use the default Drupal handlers:

```php
handlers: [
  'form' => [
    'default' => 'Drupal\Core\Entity\EntityForm',
    'delete' => 'Drupal\Core\Entity\EntityDeleteForm',
  ],
],
```

### Testing the Entity

```bash
# Clear cache to discover entity
vendor/bin/drush cache:rebuild

# Verify entity type is discovered
vendor/bin/drush php-eval "print_r(\Drupal::entityTypeManager()->getDefinition('mcp_resource_config'));"

# Create a test configuration via drush
vendor/bin/drush php-eval "
\$config = \Drupal::entityTypeManager()
  ->getStorage('mcp_resource_config')
  ->create([
    'id' => 'test_content_entity',
    'label' => 'Test Content Entity Resource',
    'resource_template_id' => 'content_entity',
    'description' => 'Test resource',
    'dependencies' => ['jsonapi'],
    'status' => TRUE,
  ]);
\$config->save();
print 'Created test config';
"

# Verify config was created
vendor/bin/drush config:get mcp_server.resource.test_content_entity

# Clean up test config
vendor/bin/drush php-eval "\Drupal::entityTypeManager()->getStorage('mcp_resource_config')->load('test_content_entity')->delete();"
```

### Key Implementation Points

1. **Readonly Properties**: Use PHP 8.1+ readonly for all entity properties
2. **Type Declarations**: Full type hints on all properties and methods
3. **Config Export**: List all properties that should be exported to config files
4. **Entity Keys**: Define id, label, and status for standard entity operations
5. **Links**: Provide all standard CRUD routes
6. **Schema Validation**: Ensure schema types match entity property types
7. **Final Class**: Entity should be final per project standards

### Configuration File Example

When saved, this entity creates a file like `config/sync/mcp_server.resource.test_content_entity.yml`:

```yaml
id: test_content_entity
label: 'Test Content Entity Resource'
resource_template_id: content_entity
description: 'Provides access to content entities'
dependencies:
  - jsonapi
status: true
```

</details>

## Implementation Notes

Follow the pattern established by McpToolConfig and McpPromptConfig entities in the codebase. Use readonly properties for all entity data and avoid getters/setters where possible per project guidelines.
