---
id: 4
group: "configuration"
dependencies: [1, 3]
status: "completed"
created: "2025-11-19"
completed: "2025-11-19"
skills:
  - drupal-forms
  - drupal-admin-ui
---
# Create Resource Management Admin UI

## Objective

Implement the administrative interface for managing MCP resource configurations, including a list builder showing all resources with dependency status and a form for creating/editing resource configurations.

## Skills Required

- **drupal-forms**: EntityForm, form API, validation, PluginSelectionFormTrait integration
- **drupal-admin-ui**: ConfigEntityListBuilder, table rendering, operations links, route definitions

## Acceptance Criteria

- [ ] List builder displays resources with columns: Label, Plugin ID, Dependencies, Status, Operations
- [ ] List builder shows visual indicators for unmet dependencies
- [ ] Form allows selection of resource template plugin via dropdown
- [ ] Form displays plugin description and dependencies (read-only)
- [ ] Form validates that required module dependencies are met before enabling
- [ ] Routes are defined for collection, add, edit, delete
- [ ] Menu links integrate with existing Tools/Prompts tabs
- [ ] Code passes PHPCS and PHPStan checks

## Technical Requirements

Create three files and update routing/menu:

1. **List Builder** (`src/McpResourceConfigListBuilder.php`):
   - Extends `ConfigEntityListBuilder`
   - Table columns: Label, Resource Type, Dependencies (with status), Enabled/Disabled, Operations
   - Override `buildHeader()` and `buildRow()`
   - Check dependency status and show warnings for unmet dependencies
   - Inject ResourceTemplateManager to access plugin definitions

2. **Configuration Form** (`src/Form/McpResourceConfigForm.php`):
   - Extends `EntityForm`
   - Uses `PluginSelectionFormTrait` for plugin selection
   - Fields: label (required), resource_template_id (select), description (read-only), dependencies (markup), status (checkbox)
   - Validate dependencies on form submission
   - Show error if required modules not enabled
   - Inject ModuleHandlerInterface for dependency checking

3. **Routes** (update `mcp_server.routing.yml`):
   - Collection route: `/admin/config/services/mcp-server/resources`
   - Add form: `/admin/config/services/mcp-server/resources/add`
   - Edit form: `/admin/config/services/mcp-server/resources/{mcp_resource_config}/edit`
   - Delete form: `/admin/config/services/mcp-server/resources/{mcp_resource_config}/delete`

4. **Menu Links** (update `mcp_server.links.menu.yml` and `mcp_server.links.task.yml`):
   - Add "Resources" tab alongside Tools and Prompts
   - Add "Add resource" action link on collection page

## Input Dependencies

- Task 1 outputs: ResourceTemplateManager for plugin discovery
- Task 3 outputs: McpResourceConfig entity
- Existing PluginSelectionFormTrait from codebase

## Output Artifacts

- `src/McpResourceConfigListBuilder.php`
- `src/Form/McpResourceConfigForm.php`
- Updated `mcp_server.routing.yml`
- Updated `mcp_server.links.menu.yml`
- Updated `mcp_server.links.task.yml`
- Functional admin UI accessible at `/admin/config/services/mcp-server/resources`

<details>
<summary>Implementation Notes</summary>

### List Builder Implementation

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\mcp_server\Entity\McpResourceConfig;
use Drupal\mcp_server\Plugin\ResourceTemplateManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of MCP Resource Configurations.
 */
final class McpResourceConfigListBuilder extends ConfigEntityListBuilder {

  /**
   * Constructs a McpResourceConfigListBuilder.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    private readonly ResourceTemplateManager $resourceTemplateManager,
    private readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($entity_type, $storage);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(
    ContainerInterface $container,
    EntityTypeInterface $entity_type,
  ): static {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('plugin.manager.mcp_resource_template'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    return [
      'label' => $this->t('Label'),
      'resource_type' => $this->t('Resource Type'),
      'dependencies' => $this->t('Dependencies'),
      'status' => $this->t('Status'),
    ] + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\mcp_server\Entity\McpResourceConfig $entity */
    $dependencies = $entity->getDependencies();
    $unmet_dependencies = array_filter(
      $dependencies,
      fn($module) => !$this->moduleHandler->moduleExists($module)
    );

    $dependency_status = '';
    if (!empty($unmet_dependencies)) {
      $dependency_status = ' ⚠️ ' . $this->t('Missing: @modules', [
        '@modules' => implode(', ', $unmet_dependencies),
      ]);
    }

    return [
      'label' => $entity->label(),
      'resource_type' => $entity->getResourceTemplateId(),
      'dependencies' => implode(', ', $dependencies) . $dependency_status,
      'status' => $entity->status() ? $this->t('Enabled') : $this->t('Disabled'),
    ] + parent::buildRow($entity);
  }

}
```

### Configuration Form Implementation

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\mcp_server\Plugin\ResourceTemplateManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for creating/editing MCP Resource Configurations.
 */
final class McpResourceConfigForm extends EntityForm {

  /**
   * Constructs a McpResourceConfigForm.
   */
  public function __construct(
    private readonly ResourceTemplateManager $resourceTemplateManager,
    private readonly ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.mcp_resource_template'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\mcp_server\Entity\McpResourceConfig $resource_config */
    $resource_config = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $resource_config->label(),
      '#description' => $this->t('Human-readable name for this resource configuration.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $resource_config->id(),
      '#machine_name' => [
        'exists' => [$this, 'exists'],
      ],
      '#disabled' => !$resource_config->isNew(),
    ];

    // Get available resource template plugins.
    $plugin_options = [];
    $definitions = $this->resourceTemplateManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      $plugin_options[$plugin_id] = $definition['label'];
    }

    $form['resource_template_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Resource Template'),
      '#options' => $plugin_options,
      '#default_value' => $resource_config->getResourceTemplateId(),
      '#required' => TRUE,
      '#description' => $this->t('Select the resource template plugin.'),
      '#ajax' => [
        'callback' => '::updatePluginFields',
        'wrapper' => 'plugin-details-wrapper',
      ],
    ];

    // Container for plugin details (description, dependencies).
    $form['plugin_details'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'plugin-details-wrapper'],
    ];

    $selected_plugin = $form_state->getValue('resource_template_id') ?? $resource_config->getResourceTemplateId();
    if ($selected_plugin && isset($definitions[$selected_plugin])) {
      $plugin_def = $definitions[$selected_plugin];

      $form['plugin_details']['description'] = [
        '#type' => 'item',
        '#title' => $this->t('Description'),
        '#markup' => $plugin_def['description'] ?? '',
      ];

      $dependencies = $plugin_def['dependencies'] ?? [];
      $unmet = array_filter($dependencies, fn($m) => !$this->moduleHandler->moduleExists($m));

      $dep_list = [];
      foreach ($dependencies as $module) {
        $status = $this->moduleHandler->moduleExists($module) ? '✓' : '✗';
        $dep_list[] = "{$status} {$module}";
      }

      $form['plugin_details']['dependencies'] = [
        '#type' => 'item',
        '#title' => $this->t('Module Dependencies'),
        '#markup' => implode('<br>', $dep_list),
      ];

      if (!empty($unmet)) {
        $form['plugin_details']['dependency_warning'] = [
          '#type' => 'item',
          '#markup' => '<div class="messages messages--warning">' .
            $this->t('⚠️ Missing required modules: @modules', [
              '@modules' => implode(', ', $unmet),
            ]) . '</div>',
        ];
      }
    }

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $resource_config->status(),
    ];

    return $form;
  }

  /**
   * AJAX callback to update plugin detail fields.
   */
  public function updatePluginFields(array &$form, FormStateInterface $form_state): array {
    return $form['plugin_details'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $plugin_id = $form_state->getValue('resource_template_id');
    $definitions = $this->resourceTemplateManager->getDefinitions();

    if (!isset($definitions[$plugin_id])) {
      $form_state->setErrorByName('resource_template_id', $this->t('Invalid resource template selected.'));
      return;
    }

    // Check dependencies if enabling.
    if ($form_state->getValue('status')) {
      $dependencies = $definitions[$plugin_id]['dependencies'] ?? [];
      $unmet = array_filter($dependencies, fn($m) => !$this->moduleHandler->moduleExists($m));

      if (!empty($unmet)) {
        $form_state->setErrorByName('status', $this->t(
          'Cannot enable resource: missing required modules (@modules)',
          ['@modules' => implode(', ', $unmet)]
        ));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    /** @var \Drupal\mcp_server\Entity\McpResourceConfig $resource_config */
    $resource_config = $this->entity;

    // Populate description and dependencies from plugin definition.
    $plugin_id = $resource_config->getResourceTemplateId();
    $definitions = $this->resourceTemplateManager->getDefinitions();
    if (isset($definitions[$plugin_id])) {
      $resource_config->description = (string) ($definitions[$plugin_id]['description'] ?? '');
      $resource_config->dependencies = $definitions[$plugin_id]['dependencies'] ?? [];
    }

    $status = $resource_config->save();
    $form_state->setRedirectUrl($resource_config->toUrl('collection'));

    return $status;
  }

  /**
   * Checks if a config entity with the given ID exists.
   */
  public function exists(string $id): bool {
    $entity = $this->entityTypeManager
      ->getStorage('mcp_resource_config')
      ->getQuery()
      ->condition('id', $id)
      ->accessCheck(FALSE)
      ->execute();
    return !empty($entity);
  }

}
```

### Routes Configuration

Add to `mcp_server.routing.yml`:

```yaml
entity.mcp_resource_config.collection:
  path: '/admin/config/services/mcp-server/resources'
  defaults:
    _entity_list: 'mcp_resource_config'
    _title: 'MCP Resource Templates'
  requirements:
    _permission: 'administer mcp server'

entity.mcp_resource_config.add_form:
  path: '/admin/config/services/mcp-server/resources/add'
  defaults:
    _entity_form: 'mcp_resource_config.add'
    _title: 'Add Resource Configuration'
  requirements:
    _entity_create_access: 'mcp_resource_config'

entity.mcp_resource_config.edit_form:
  path: '/admin/config/services/mcp-server/resources/{mcp_resource_config}/edit'
  defaults:
    _entity_form: 'mcp_resource_config.edit'
    _title: 'Edit Resource Configuration'
  requirements:
    _entity_access: 'mcp_resource_config.update'

entity.mcp_resource_config.delete_form:
  path: '/admin/config/services/mcp-server/resources/{mcp_resource_config}/delete'
  defaults:
    _entity_form: 'mcp_resource_config.delete'
    _title: 'Delete Resource Configuration'
  requirements:
    _entity_access: 'mcp_resource_config.delete'
```

### Menu/Task Links

Add to `mcp_server.links.task.yml`:

```yaml
entity.mcp_resource_config.collection:
  route_name: entity.mcp_resource_config.collection
  base_route: mcp_server.settings
  title: 'Resources'
  weight: 30
```

Add to `mcp_server.links.action.yml` (create if doesn't exist):

```yaml
entity.mcp_resource_config.add_form:
  route_name: entity.mcp_resource_config.add_form
  title: 'Add resource'
  appears_on:
    - entity.mcp_resource_config.collection
```

### Testing the UI

```bash
# Clear cache
vendor/bin/drush cache:rebuild

# Navigate to admin UI
# Visit: /admin/config/services/mcp-server/resources

# Or test via drush
vendor/bin/drush php-eval "
\$url = \Drupal\Core\Url::fromRoute('entity.mcp_resource_config.collection')->toString();
print 'Visit: ' . \$url;
"
```

</details>

## Implementation Notes

Follow the established patterns from McpToolConfigListBuilder and McpToolConfigForm. Use AJAX to dynamically update dependency information when plugin selection changes. Ensure dependency validation prevents enabling resources when required modules are missing.
