---
id: 5
group: "mcp-integration"
dependencies: [1, 2, 3]
status: "completed"
created: "2025-11-19"
skills:
  - drupal-services
  - mcp-sdk
---
# Implement Resource Bridge Service and MCP Server Integration

## Objective

Create the McpResourceBridgeService that loads enabled resource configurations and integrates them with the MCP server. Update McpServerFactory to register resources with the MCP server instance, making them discoverable and accessible to MCP clients.

## Skills Required

- **drupal-services**: Service definition, dependency injection, entity queries, caching
- **mcp-sdk**: Understanding MCP server builder API, resource registration, handler callbacks

## Acceptance Criteria

- [ ] McpResourceBridgeService is created with methods: getEnabledResources(), getResourceContent(), checkResourceAccess()
- [ ] Service loads enabled McpResourceConfig entities and queries their plugins
- [ ] Service implements caching with appropriate cache tags
- [ ] McpServerFactory::create() registers resources via $builder->addResource()
- [ ] Service definition added to mcp_server.services.yml
- [ ] Resources are discoverable by MCP clients
- [ ] Code passes PHPCS and PHPStan checks

## Technical Requirements

Create/update three files:

1. **Bridge Service** (`src/McpResourceBridgeService.php`):
   - Similar to McpBridgeService pattern
   - Methods:
     - `getEnabledResources(): array` - Returns resource metadata for all enabled configs
     - `getResourceContent(string $uri): ?array` - Fetches content via plugin
     - `checkResourceAccess(string $uri, AccountInterface $account): AccessResultInterface` - Checks access via plugin
   - Inject: EntityTypeManager, ResourceTemplateManager, LoggerChannel, CacheBackend, CurrentUser
   - Cache results with 'mcp_server:discovery' cache tag

2. **Server Factory Update** (`src/McpServerFactory.php`):
   - Inject McpResourceBridgeService
   - In `create()` method, after tool/prompt registration, register resources
   - Call `$builder->addResource()` for each enabled resource
   - Handle exceptions gracefully with logging

3. **Service Definition** (update `mcp_server.services.yml`):
   - Define `mcp_server.resource_bridge` service
   - Add dependency to McpServerFactory

## Input Dependencies

- Task 1 outputs: ResourceTemplateManager service
- Task 2 outputs: ContentEntityResourceTemplate plugin
- Task 3 outputs: McpResourceConfig entity
- Existing McpBridgeService (for pattern reference)

## Output Artifacts

- `src/McpResourceBridgeService.php`
- Updated `src/McpServerFactory.php`
- Updated `mcp_server.services.yml`
- Resources registered with MCP server and accessible via MCP protocol

<details>
<summary>Implementation Notes</summary>

### Bridge Service Implementation

```php
<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\mcp_server\Entity\McpResourceConfig;
use Drupal\mcp_server\Plugin\ResourceTemplateManager;

/**
 * Bridge service integrating Resource plugins with MCP configurations.
 */
final class McpResourceBridgeService {

  /**
   * Constructs a McpResourceBridgeService.
   */
  public function __construct(
    private readonly ResourceTemplateManager $resourceTemplateManager,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerChannelInterface $logger,
    private readonly AccountProxyInterface $currentUser,
    private readonly CacheBackendInterface $cache,
  ) {}

  /**
   * Gets all enabled MCP resources.
   *
   * @return array<int, array>
   *   Array of resource metadata. Each resource contains:
   *   - uri: string - The resource URI template
   *   - name: string - Human-readable name
   *   - description: string - Resource description
   *   - mimeType: string - MIME type
   */
  public function getEnabledResources(): array {
    $cache_id = 'mcp_server:enabled_resources';
    $cached = $this->cache->get($cache_id);

    if ($cached !== FALSE) {
      return $cached->data;
    }

    try {
      $storage = $this->entityTypeManager->getStorage('mcp_resource_config');
      $query = $storage->getQuery()
        ->condition('status', TRUE)
        ->accessCheck(FALSE);

      $config_ids = $query->execute();
      $configs = $storage->loadMultiple($config_ids);
      $enabled_resources = [];

      foreach ($configs as $config) {
        if (!$config instanceof McpResourceConfig) {
          continue;
        }

        $plugin_id = $config->getResourceTemplateId();

        try {
          /** @var \Drupal\mcp_server\Plugin\ResourceTemplateInterface $plugin */
          $plugin = $this->resourceTemplateManager->createInstance($plugin_id);
          $resources = $plugin->getResources();

          foreach ($resources as $resource) {
            $enabled_resources[] = $resource;
          }
        }
        catch (\Exception $e) {
          $this->logger->error(
            'Failed to load resource plugin @plugin_id: @message',
            [
              '@plugin_id' => $plugin_id,
              '@message' => $e->getMessage(),
            ]
          );
        }
      }

      $this->cache->set(
        $cache_id,
        $enabled_resources,
        CacheBackendInterface::CACHE_PERMANENT,
        ['mcp_server:discovery']
      );

      return $enabled_resources;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve enabled resources: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

  /**
   * Gets content for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return array|null
   *   Resource content array, or NULL if not found/accessible.
   */
  public function getResourceContent(string $uri): ?array {
    // Find which plugin handles this URI.
    $plugin = $this->findPluginForUri($uri);
    if ($plugin === NULL) {
      return NULL;
    }

    // Check access first.
    $access = $plugin->checkAccess($uri, $this->currentUser->getAccount());
    if (!$access->isAllowed()) {
      return NULL;
    }

    // Get content from plugin.
    try {
      return $plugin->getResourceContent($uri);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to get resource content for @uri: @message',
        [
          '@uri' => $uri,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Checks access for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function checkResourceAccess(
    string $uri,
    \Drupal\Core\Session\AccountInterface $account,
  ): AccessResultInterface {
    $plugin = $this->findPluginForUri($uri);
    if ($plugin === NULL) {
      return \Drupal\Core\Access\AccessResult::forbidden('No plugin found for URI');
    }

    try {
      return $plugin->checkAccess($uri, $account);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to check access for @uri: @message',
        [
          '@uri' => $uri,
          '@message' => $e->getMessage(),
        ]
      );
      return \Drupal\Core\Access\AccessResult::forbidden('Access check failed');
    }
  }

  /**
   * Finds the plugin instance that handles a given URI.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return \Drupal\mcp_server\Plugin\ResourceTemplateInterface|null
   *   Plugin instance, or NULL if not found.
   */
  private function findPluginForUri(string $uri): ?\Drupal\mcp_server\Plugin\ResourceTemplateInterface {
    try {
      $storage = $this->entityTypeManager->getStorage('mcp_resource_config');
      $query = $storage->getQuery()
        ->condition('status', TRUE)
        ->accessCheck(FALSE);

      $config_ids = $query->execute();
      $configs = $storage->loadMultiple($config_ids);

      foreach ($configs as $config) {
        if (!$config instanceof McpResourceConfig) {
          continue;
        }

        $plugin_id = $config->getResourceTemplateId();

        try {
          $plugin = $this->resourceTemplateManager->createInstance($plugin_id);
          // Check if this plugin's URI template matches.
          // For now, we check by attempting to get content.
          // A more sophisticated approach would parse URI templates.
          $resources = $plugin->getResources();
          foreach ($resources as $resource) {
            // Simple URI matching - improve this for production.
            if (str_starts_with($uri, 'drupal://entity/')) {
              return $plugin;
            }
          }
        }
        catch (\Exception $e) {
          continue;
        }
      }

      return NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

}
```

### Server Factory Integration

Update `src/McpServerFactory.php` constructor:

```php
public function __construct(
  private readonly ConfigFactoryInterface $configFactory,
  private readonly McpBridgeService $mcpBridge,
  private readonly LoggerInterface $logger,
  private readonly EventDispatcherInterface $eventDispatcher,
  private readonly FileSessionStore $sessionStore,
  private readonly PromptArgumentCompletionProviderManager $completionProviderManager,
  private readonly McpResourceBridgeService $resourceBridge,  // ADD THIS
) {}
```

Update `create()` method in `McpServerFactory.php`, after prompt registration:

```php
// Register prompts using custom loader.
$promptLoader = new PromptConfigLoader(
  $this->mcpBridge,
  $this->logger,
  $this->completionProviderManager
);
$builder->addLoaders($promptLoader);

// Register resources (NEW CODE).
try {
  $enabled_resources = $this->resourceBridge->getEnabledResources();
  foreach ($enabled_resources as $resource_data) {
    $builder->addResource(
      handler: fn(string $uri) => $this->resourceBridge->getResourceContent($uri),
      uri: $resource_data['uri'],
      name: $resource_data['name'],
      description: $resource_data['description'] ?? '',
      mimeType: $resource_data['mimeType'] ?? 'application/json',
    );
  }
} catch (\Exception $e) {
  $this->logger->error(
    'Failed to register MCP resources: @message',
    ['@message' => $e->getMessage()]
  );
}

return $builder->build();
```

### Service Definition

Add to `mcp_server.services.yml`:

```yaml
  mcp_server.resource_bridge:
    class: Drupal\mcp_server\McpResourceBridgeService
    arguments:
      - '@plugin.manager.mcp_resource_template'
      - '@entity_type.manager'
      - '@logger.channel.mcp_server'
      - '@current_user'
      - '@cache.default'
```

Update `mcp_server.server_factory`:

```yaml
  mcp_server.server_factory:
    class: Drupal\mcp_server\McpServerFactory
    arguments:
      - '@config.factory'
      - '@mcp_server.bridge'
      - '@logger.channel.mcp_server'
      - '@event_dispatcher'
      - '@mcp_server.session_store'
      - '@plugin.manager.mcp_prompt_argument_completion_provider'
      - '@mcp_server.resource_bridge'  # ADD THIS
```

### Testing Integration

```bash
# Clear cache
vendor/bin/drush cache:rebuild

# Create a test resource config
vendor/bin/drush php-eval "
\$config = \Drupal::entityTypeManager()
  ->getStorage('mcp_resource_config')
  ->create([
    'id' => 'content_entity_node',
    'label' => 'Node Resources',
    'resource_template_id' => 'content_entity',
    'status' => TRUE,
  ]);
\$config->save();
print 'Created resource config';
"

# Test bridge service
vendor/bin/drush php-eval "
\$bridge = \Drupal::service('mcp_server.resource_bridge');
\$resources = \$bridge->getEnabledResources();
print 'Found ' . count(\$resources) . ' resources';
print_r(\$resources);
"

# Create test node and access via resource
vendor/bin/drush php-eval "
\$node = \Drupal\node\Entity\Node::create([
  'type' => 'page',
  'title' => 'Test Node',
  'status' => 1,
]);
\$node->save();
\$nid = \$node->id();

\$bridge = \Drupal::service('mcp_server.resource_bridge');
\$content = \$bridge->getResourceContent('drupal://entity/node/' . \$nid);
print_r(\$content);
"
```

### Key Implementation Points

1. **Caching**: Cache enabled resources to avoid repeated config entity queries
2. **Error Handling**: Wrap plugin instantiation in try-catch blocks
3. **Logging**: Log errors for debugging but don't expose to MCP clients
4. **Access Control**: Always check access before returning content
5. **URI Matching**: Improve `findPluginForUri()` for production (use proper URI template matching)
6. **MCP SDK API**: Verify `$builder->addResource()` signature matches current MCP SDK version

</details>

## Implementation Notes

Follow the pattern from McpBridgeService for tools/prompts. Ensure the MCP SDK's resource registration API is properly understood - review MCP SDK documentation if the API differs from expectations.
