---
id: 6
group: "testing"
dependencies: [1, 2, 3, 4, 5]
status: "completed"
created: "2025-11-19"
completed: "2025-11-19"
skills:
  - phpunit
  - drupal-testing
---
# Implement Integration Tests for Resource System

## Objective

Create a comprehensive kernel test suite that validates the entire resource system including plugin discovery, entity access control, JSON:API serialization, configuration management, and end-to-end resource access flow. This single consolidated test class covers all critical functionality per project testing guidelines.

## Skills Required

- **phpunit**: Test structure, assertions, data providers, helper methods
- **drupal-testing**: KernelTestBase, test entities, test users, module installation in tests

## Acceptance Criteria

- [ ] Single kernel test class covers all resource system functionality
- [ ] Tests validate plugin discovery and instantiation
- [ ] Tests verify content entity discovery (finds node, media, taxonomy_term)
- [ ] Tests enforce access control (published vs unpublished, anonymous vs authenticated)
- [ ] Tests verify JSON:API serialization produces valid structure
- [ ] Tests validate configuration entity CRUD operations
- [ ] Tests check dependency validation (JSON:API module requirement)
- [ ] All tests pass via `vendor/bin/phpunit`
- [ ] Code coverage includes critical paths: access control, serialization, plugin system
- [ ] Code passes PHPCS and PHPStan checks

## Technical Requirements

Create `tests/src/Kernel/ResourceTemplateSystemTest.php`:

**IMPORTANT: Meaningful Test Strategy Guidelines**

Your critical mantra for test generation is: "write a few tests, mostly integration".

**Definition of "Meaningful Tests":**
Tests that verify custom business logic, critical paths, and edge cases specific to the application. Focus on testing YOUR code, not the framework or library functionality.

**When TO Write Tests:**
- Custom business logic and algorithms
- Critical user workflows and data transformations
- Edge cases and error conditions for core functionality
- Integration points between different system components
- Complex validation logic or calculations

**When NOT to Write Tests:**
- Third-party library functionality (already tested upstream)
- Framework features (React hooks, Express middleware, etc.)
- Simple CRUD operations without custom logic
- Getter/setter methods or basic property access
- Configuration files or static data
- Obvious functionality that would break immediately if incorrect

**Test Task Creation Rules:**
- Combine related test scenarios into single tasks (e.g., "Test user authentication flow" not separate tasks for login, logout, validation)
- Focus on integration and critical path testing over unit test coverage
- Avoid creating separate tasks for testing each CRUD operation individually
- Question whether simple functions need dedicated test tasks

**Test Scenarios** (use helper methods, not separate test methods):

1. **Plugin System**:
   - ResourceTemplateManager discovers ContentEntityResourceTemplate plugin
   - Plugin instantiation works and returns valid instance
   - Plugin metadata (id, label, description, dependencies) is accessible

2. **Content Entity Discovery**:
   - Plugin discovers node, media, taxonomy_term entity types
   - Plugin excludes non-content entities or entities without canonical URLs
   - Plugin generates correct URI templates for each entity type

3. **Entity Loading and Access Control**:
   - Published node accessible to anonymous users
   - Unpublished node NOT accessible to anonymous users
   - Unpublished node accessible to user with 'view any unpublished content' permission
   - Non-existent entity returns NULL content and forbidden access
   - Access result preserves cache metadata

4. **JSON:API Serialization**:
   - Serialized entity has 'data', 'attributes', 'links' keys
   - Entity fields are properly serialized
   - Entity relationships are included
   - Field-level access is respected (sensitive fields filtered)

5. **Configuration Entity**:
   - McpResourceConfig can be created, saved, loaded
   - Entity exports to config with correct schema
   - Entity properties (resource_template_id, dependencies) are accessible

6. **Dependency Validation**:
   - Form validation fails if JSON:API module is disabled and resource enabled
   - List builder shows warning for unmet dependencies

7. **Bridge Service**:
   - getEnabledResources() returns resources from enabled configs only
   - getResourceContent() returns JSON:API serialized content
   - checkResourceAccess() enforces entity access control
   - Service caching works correctly

8. **End-to-End Integration**:
   - Create resource config → Enable → Access via bridge service → Get JSON:API content
   - Disable resource config → Resources no longer returned by getEnabledResources()

**Test Data Setup**:
- Install modules: node, media, taxonomy, jsonapi, user
- Create content types: page, article
- Create users: anonymous, authenticated without permissions, editor with 'view any unpublished content'
- Create test entities: published node, unpublished node, media entity, taxonomy term
- Create test resource configs: enabled content_entity config, disabled config

**Testing Constraints** (per project guidelines):
- ONE kernel test class for the entire feature
- Use helper methods to organize test scenarios
- Test module-specific logic, not framework features
- Don't test Drupal's entity loading system, test YOUR access control logic
- Don't test JSON:API module, test that you're calling it correctly

## Input Dependencies

- All previous tasks (1-5) completed
- ContentEntityResourceTemplate plugin
- McpResourceConfig entity
- McpResourceBridgeService
- Plugin manager service

## Output Artifacts

- `tests/src/Kernel/ResourceTemplateSystemTest.php`
- Passing test suite validating all critical functionality
- Test coverage report showing resource system is properly tested

<details>
<summary>Implementation Notes</summary>

### Test Class Structure

```php
<?php

declare(strict_types=1);

namespace Drupal\Tests\mcp_server\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests the MCP Resource Template system.
 *
 * @group mcp_server
 * @coversDefaultClass \Drupal\mcp_server\Plugin\ResourceTemplate\ContentEntityResourceTemplate
 */
final class ResourceTemplateSystemTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'media',
    'taxonomy',
    'jsonapi',
    'serialization',
    'mcp_server',
  ];

  /**
   * The resource template manager.
   */
  private $resourceTemplateManager;

  /**
   * The resource bridge service.
   */
  private $resourceBridge;

  /**
   * Test user with view unpublished permission.
   */
  private $editorUser;

  /**
   * Test user without special permissions.
   */
  private $authenticatedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'jsonapi']);

    // Create content type.
    NodeType::create([
      'type' => 'page',
      'name' => 'Basic page',
    ])->save();

    // Create test users.
    $this->authenticatedUser = User::create([
      'name' => 'authenticated_user',
      'status' => 1,
    ]);
    $this->authenticatedUser->save();

    // Create editor role with view unpublished permission.
    $editor_role = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
    ]);
    $editor_role->grantPermission('view any unpublished content');
    $editor_role->save();

    $this->editorUser = User::create([
      'name' => 'editor_user',
      'status' => 1,
    ]);
    $this->editorUser->addRole('editor');
    $this->editorUser->save();

    // Get services.
    $this->resourceTemplateManager = $this->container->get('plugin.manager.mcp_resource_template');
    $this->resourceBridge = $this->container->get('mcp_server.resource_bridge');
  }

  /**
   * Tests the complete resource system integration.
   */
  public function testResourceSystemIntegration(): void {
    // Test plugin discovery.
    $this->assertPluginDiscovery();

    // Test entity discovery.
    $this->assertEntityDiscovery();

    // Test access control.
    $this->assertAccessControl();

    // Test JSON:API serialization.
    $this->assertJsonApiSerialization();

    // Test configuration entity.
    $this->assertConfigurationEntity();

    // Test bridge service.
    $this->assertBridgeService();

    // Test end-to-end flow.
    $this->assertEndToEndFlow();
  }

  /**
   * Helper: Tests plugin discovery.
   */
  private function assertPluginDiscovery(): void {
    $definitions = $this->resourceTemplateManager->getDefinitions();

    $this->assertArrayHasKey('content_entity', $definitions, 'ContentEntityResourceTemplate plugin is discovered');

    $plugin_def = $definitions['content_entity'];
    $this->assertSame('content_entity', $plugin_def['id']);
    $this->assertArrayHasKey('label', $plugin_def);
    $this->assertArrayHasKey('dependencies', $plugin_def);
    $this->assertContains('jsonapi', $plugin_def['dependencies']);

    // Instantiate plugin.
    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $this->assertInstanceOf(
      'Drupal\mcp_server\Plugin\ResourceTemplateInterface',
      $plugin,
      'Plugin implements ResourceTemplateInterface'
    );
  }

  /**
   * Helper: Tests entity type discovery.
   */
  private function assertEntityDiscovery(): void {
    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $resources = $plugin->getResources();

    $this->assertNotEmpty($resources, 'Plugin discovers content entities');

    // Extract entity types from resources.
    $entity_types = array_map(function($resource) {
      if (preg_match('#drupal://entity/([^/]+)/#', $resource['uri'], $matches)) {
        return $matches[1];
      }
      return NULL;
    }, $resources);
    $entity_types = array_filter($entity_types);

    $this->assertContains('node', $entity_types, 'Node entity type discovered');

    // Verify URI format.
    $node_resource = array_filter($resources, fn($r) => str_contains($r['uri'], 'drupal://entity/node/'));
    $node_resource = reset($node_resource);
    $this->assertStringContainsString('drupal://entity/node/', $node_resource['uri']);
    $this->assertSame('application/vnd.api+json', $node_resource['mimeType']);
  }

  /**
   * Helper: Tests access control.
   */
  private function assertAccessControl(): void {
    // Create published and unpublished nodes.
    $published_node = Node::create([
      'type' => 'page',
      'title' => 'Published Node',
      'status' => 1,
    ]);
    $published_node->save();

    $unpublished_node = Node::create([
      'type' => 'page',
      'title' => 'Unpublished Node',
      'status' => 0,
    ]);
    $unpublished_node->save();

    $plugin = $this->resourceTemplateManager->createInstance('content_entity');

    // Anonymous user can access published.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$published_node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'Anonymous user can access published node');

    // Anonymous user cannot access unpublished.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$unpublished_node->id()}",
      User::getAnonymousUser()
    );
    $this->assertFalse($access->isAllowed(), 'Anonymous user cannot access unpublished node');

    // Editor can access unpublished.
    $access = $plugin->checkAccess(
      "drupal://entity/node/{$unpublished_node->id()}",
      $this->editorUser
    );
    $this->assertTrue($access->isAllowed(), 'Editor can access unpublished node');

    // Non-existent entity.
    $access = $plugin->checkAccess('drupal://entity/node/99999', User::getAnonymousUser());
    $this->assertFalse($access->isAllowed(), 'Non-existent entity returns forbidden');
  }

  /**
   * Helper: Tests JSON:API serialization.
   */
  private function assertJsonApiSerialization(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test JSON:API Node',
      'status' => 1,
    ]);
    $node->save();

    $plugin = $this->resourceTemplateManager->createInstance('content_entity');
    $content = $plugin->getResourceContent("drupal://entity/node/{$node->id()}");

    $this->assertIsArray($content, 'Resource content is array');
    $this->assertArrayHasKey('data', $content, 'JSON:API response has data key');

    $data = $content['data'];
    $this->assertArrayHasKey('type', $data);
    $this->assertArrayHasKey('id', $data);
    $this->assertArrayHasKey('attributes', $data);
    $this->assertSame('node--page', $data['type']);
    $this->assertArrayHasKey('title', $data['attributes']);
  }

  /**
   * Helper: Tests configuration entity CRUD.
   */
  private function assertConfigurationEntity(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('mcp_resource_config');

    // Create config.
    $config = $storage->create([
      'id' => 'test_content_entity',
      'label' => 'Test Content Entity',
      'resource_template_id' => 'content_entity',
      'status' => TRUE,
    ]);
    $config->save();

    // Load config.
    $loaded = $storage->load('test_content_entity');
    $this->assertNotNull($loaded, 'Configuration entity saved and loaded');
    $this->assertSame('content_entity', $loaded->getResourceTemplateId());
    $this->assertTrue($loaded->status());

    // Update config.
    $loaded->set('status', FALSE);
    $loaded->save();
    $reloaded = $storage->load('test_content_entity');
    $this->assertFalse($reloaded->status());

    // Delete config.
    $loaded->delete();
    $deleted = $storage->load('test_content_entity');
    $this->assertNull($deleted, 'Configuration entity deleted');
  }

  /**
   * Helper: Tests bridge service.
   */
  private function assertBridgeService(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('mcp_resource_config');

    // Create enabled config.
    $config = $storage->create([
      'id' => 'enabled_content_entity',
      'label' => 'Enabled Content Entity',
      'resource_template_id' => 'content_entity',
      'status' => TRUE,
    ]);
    $config->save();

    // Clear cache.
    $this->container->get('cache.default')->deleteAll();

    // Get enabled resources.
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertNotEmpty($resources, 'Bridge service returns enabled resources');

    // Create test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Bridge Test Node',
      'status' => 1,
    ]);
    $node->save();

    // Get resource content via bridge.
    $content = $this->resourceBridge->getResourceContent("drupal://entity/node/{$node->id()}");
    $this->assertIsArray($content, 'Bridge service returns content');
    $this->assertArrayHasKey('data', $content);

    // Check access via bridge.
    $access = $this->resourceBridge->checkResourceAccess(
      "drupal://entity/node/{$node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'Bridge service checks access');

    // Disable config.
    $config->set('status', FALSE);
    $config->save();
    $this->container->get('cache.default')->deleteAll();

    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertEmpty($resources, 'Disabled resources not returned');

    $config->delete();
  }

  /**
   * Helper: Tests complete end-to-end flow.
   */
  private function assertEndToEndFlow(): void {
    // 1. Create resource configuration.
    $storage = $this->container->get('entity_type.manager')->getStorage('mcp_resource_config');
    $config = $storage->create([
      'id' => 'e2e_test',
      'label' => 'E2E Test Resource',
      'resource_template_id' => 'content_entity',
      'status' => TRUE,
    ]);
    $config->save();

    // 2. Create test content.
    $node = Node::create([
      'type' => 'page',
      'title' => 'E2E Test Node',
      'status' => 1,
    ]);
    $node->save();

    // 3. Clear cache.
    $this->container->get('cache.default')->deleteAll();

    // 4. Get enabled resources via bridge.
    $resources = $this->resourceBridge->getEnabledResources();
    $this->assertNotEmpty($resources, 'E2E: Resources enabled');

    // 5. Access resource content.
    $content = $this->resourceBridge->getResourceContent("drupal://entity/node/{$node->id()}");
    $this->assertIsArray($content, 'E2E: Content retrieved');
    $this->assertArrayHasKey('data', $content);
    $this->assertSame('E2E Test Node', $content['data']['attributes']['title']);

    // 6. Verify access control.
    $access = $this->resourceBridge->checkResourceAccess(
      "drupal://entity/node/{$node->id()}",
      User::getAnonymousUser()
    );
    $this->assertTrue($access->isAllowed(), 'E2E: Access control enforced');

    // Cleanup.
    $config->delete();
  }

}
```

### Running the Tests

```bash
# Run all mcp_server tests
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/

# Run specific test class
vendor/bin/phpunit --filter ResourceTemplateSystemTest web/modules/contrib/mcp_server/tests/

# Run with coverage (if xdebug enabled)
vendor/bin/phpunit --coverage-html coverage web/modules/contrib/mcp_server/tests/
```

### Key Testing Points

1. **Helper Methods**: Organize related test scenarios into helper methods, not separate test methods
2. **Test Data**: Create minimal test data needed for validation
3. **Module Installation**: Install only required modules (node, jsonapi, user, mcp_server)
4. **Cache Clearing**: Clear cache when testing configuration changes
5. **Assertions**: Use specific assertions (assertArrayHasKey, assertContains, etc.)
6. **Access Control**: Test multiple user scenarios (anonymous, authenticated, editor)
7. **Error Cases**: Test non-existent entities, invalid URIs, disabled modules
8. **Integration**: Test the full flow from config creation to content retrieval

</details>

## Implementation Notes

This consolidated test class follows project guidelines: ONE kernel test per feature, using helper methods to organize scenarios. Focus on testing module-specific logic (access control integration, plugin discovery, bridge service) rather than framework functionality.
