---
id: 14
summary: "Add configurable entity filtering to ContentEntityResourceTemplate plugin with URL requirement checkbox and entity type deny list"
created: 2025-11-20
---

# Plan: Configurable Entity Filtering for Content Entity Resources

## Original Work Order

> I want to update the resource plugin for entities. I want to add a checkbox in the plugin configuration form that make "Entity Type Has URL" instead of hardcoding the $entity_type->hasLinkTemplate('canonical'). In addition to that, I want to add checkboxes per entity type as a deny list. Entity types selected on that deny list will not be exposed as resource templates.

## Executive Summary

This plan enhances the ContentEntityResourceTemplate plugin by replacing hardcoded entity filtering logic with configurable options accessible through the plugin's settings form. Currently, the plugin hardcodes the requirement that entities must have a canonical link template (`hasLinkTemplate('canonical')`) to be exposed. This plan introduces two configuration options: (1) a checkbox to toggle the URL requirement, allowing entities without canonical URLs to be exposed if desired, and (2) a deny list using checkboxes for each entity type, giving administrators granular control over which entity types are excluded from MCP resource exposure.

The implementation leverages the existing plugin configuration system (`buildConfigurationForm`, `defaultConfiguration`, `submitConfigurationForm`) in ResourceTemplateBase, requiring minimal changes to the architecture while providing significant flexibility.

## Context

### Current State

The `ContentEntityResourceTemplate` plugin in `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php` currently:
- Hardcodes the requirement `$entity_type->hasLinkTemplate('canonical')` at line 115
- Filters entities based only on whether they implement `ContentEntityInterface` and have canonical URLs
- Provides no configuration options in its form (only inherits the default `enabled` checkbox from base class)
- Automatically exposes all content entities with canonical URLs as MCP resources

### Target State

After implementation:
- Plugin configuration form displays a checkbox labeled "Require Canonical URL" (defaulting to enabled for backward compatibility)
- Plugin configuration form displays checkboxes for each available content entity type in a deny list fieldset
- The `getResources()` and `getResourceContent()` methods respect both configuration options
- Configuration is stored in `mcp_server.resource_plugins` config under the plugin's configuration array
- Entity filtering logic becomes transparent and administrator-controllable

### Background

The MCP Server module uses a plugin-based architecture for resource templates. Each plugin extends `ResourceTemplateBase`, which provides standard methods like `buildConfigurationForm()` for plugin-specific settings. The form is integrated into `ResourcePluginSettingsForm`, which renders all plugin configurations in a unified interface. The current hardcoded filtering was appropriate for initial implementation but limits flexibility for sites with custom entity types or unusual URL patterns.

## Technical Implementation Approach

```mermaid
flowchart TD
    A[Plugin Configuration] --> B[defaultConfiguration]
    A --> C[buildConfigurationForm]
    A --> D[submitConfigurationForm]

    B --> E[Set defaults:<br/>require_canonical_url: true<br/>denied_entity_types: empty]

    C --> F[Add URL requirement checkbox]
    C --> G[Add entity type deny list]

    G --> H[Query all content entity types]
    H --> I[Generate checkboxes fieldset]

    D --> J[Process form values]
    J --> K[Store in configuration array]

    K --> L[getResources uses config]
    K --> M[getResourceContent uses config]

    L --> N{require_canonical_url?}
    N -->|Yes| O[Check hasLinkTemplate]
    N -->|No| P[Skip URL check]

    L --> Q{Entity in deny list?}
    Q -->|Yes| R[Exclude entity]
    Q -->|No| S[Include entity]
```

### Configuration Storage

Configuration will be stored in the plugin's configuration array as follows:

```php
[
  'id' => 'content_entity',
  'enabled' => TRUE,
  'configuration' => [
    'require_canonical_url' => TRUE,  // Checkbox value
    'denied_entity_types' => [        // Array of denied entity type IDs
      'user' => 'user',
      'shortcut' => 'shortcut',
    ],
  ],
]
```

### Form Implementation

Override `defaultConfiguration()` in `ContentEntityResourceTemplate`:
```php
protected function defaultConfiguration(): array {
  return parent::defaultConfiguration() + [
    'require_canonical_url' => TRUE,
    'denied_entity_types' => [],
  ];
}
```

Override `buildConfigurationForm()` to add form elements:
- Single checkbox for "Require entities to have canonical URL"
- Checkboxes fieldset listing all content entity types by label, allowing multiple selections for deny list

Override `submitConfigurationForm()` to process submitted values into configuration array format.

### Resource Filtering Implementation

Update `getResources()` method (around line 103-135):
1. Check `$this->configuration['require_canonical_url']` before applying `hasLinkTemplate('canonical')` check
2. Check if entity type ID exists in `$this->configuration['denied_entity_types']` array before including

Update `getResourceContent()` and `checkAccess()` methods (lines 140-252) to apply the same filtering logic for consistency.

## Risk Considerations and Mitigation Strategies

### Technical Risks

- **Backward Compatibility**: Changing from hardcoded to configurable filtering might expose unexpected entity types
    - **Mitigation**: Default `require_canonical_url` to `TRUE` to maintain current behavior; deny list defaults to empty

- **Form State Handling**: Checkbox fieldset values need proper processing to convert to array format
    - **Mitigation**: Use `array_filter()` in `submitConfigurationForm()` to remove unchecked items, ensuring clean array storage

### Implementation Risks

- **Configuration Validation**: Invalid entity type IDs in deny list could cause errors
    - **Mitigation**: Validate deny list entries against `EntityTypeManager::getDefinitions()` in `submitConfigurationForm()`

- **Performance**: Loading all entity type definitions in form might impact admin UI
    - **Mitigation**: Entity type definitions are cached by Drupal core; minimal performance impact expected

## Success Criteria

### Primary Success Criteria

1. Plugin configuration form displays "Require Canonical URL" checkbox, defaulting to checked
2. Plugin configuration form displays checkboxes for all content entity types under "Denied Entity Types" fieldset
3. Unchecking "Require Canonical URL" allows entities without canonical URLs to be exposed as resources
4. Entities selected in deny list are excluded from `getResources()`, `getResourceContent()`, and `checkAccess()`
5. Configuration persists correctly in `mcp_server.resource_plugins` config
6. All existing kernel tests pass without modification

### Quality Assurance Metrics

1. Code follows PHP 8.3+ standards with strict types and readonly properties where appropriate
2. PHPStan analysis passes with no new errors
3. New Kernel test validates configuration storage and filtering logic
4. Manual testing confirms UI displays correctly in resource plugin settings form

## Resource Requirements

### Development Skills

- Drupal plugin system and configuration management
- Drupal Form API with checkboxes and fieldsets
- PHP array manipulation and filtering
- Understanding of Drupal entity system

### Technical Infrastructure

- Existing ContentEntityResourceTemplate plugin
- Existing ResourcePluginSettingsForm integration
- Drupal entity type manager service
- Kernel testing framework

## Notes

- This implementation does not require schema changes or database updates
- The configuration is stored in exportable config, allowing site administrators to manage via CMI
- No changes needed to MCP protocol implementation or JSON:API serialization

## Task Dependencies

```mermaid
graph TD
    001[Task 001: Implement Configuration Form] --> 002[Task 002: Update Filtering Logic]
```

## Execution Blueprint

**Validation Gates:**
- Reference: `/config/hooks/POST_PHASE.md`

### Phase 1: Configuration Infrastructure
**Parallel Tasks:**
- Task 001: Implement Configuration Form Methods

**Objectives**: Establish the configuration form methods (`defaultConfiguration`, `buildConfigurationForm`, `submitConfigurationForm`) that provide the UI and data handling for the new filtering options.

### Phase 2: Apply Configuration Logic
**Parallel Tasks:**
- Task 002: Update Filtering Logic to Use Configuration (depends on: 001)

**Objectives**: Integrate the configuration values into the entity filtering logic across all three methods (`getResources`, `getResourceContent`, `checkAccess`).

### Execution Summary
- Total Phases: 2
- Total Tasks: 2
- Maximum Parallelism: 1 task per phase
- Critical Path Length: 2 phases

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-20

### Results

Successfully implemented configurable entity filtering for the ContentEntityResourceTemplate plugin. The implementation adds two new configuration options accessible through the plugin's settings form:

1. **"Require Canonical URL" checkbox**: Allows administrators to toggle whether entities must have canonical link templates to be exposed as MCP resources (defaults to TRUE for backward compatibility)

2. **Entity Type Deny List**: Checkboxes fieldset allowing administrators to select specific content entity types to exclude from MCP resource exposure

**Deliverables:**
- Added 3 configuration methods to `ContentEntityResourceTemplate.php`: `defaultConfiguration()`, `buildConfigurationForm()`, `submitConfigurationForm()`
- Updated filtering logic in 3 methods: `getResources()`, `getResourceContent()`, `checkAccess()`
- All existing tests pass (37 tests)
- PHPStan analysis clean with no new errors
- Configuration properly persists in `mcp_server.resource_plugins` config

**Files Modified:**
- `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php` (added ~80 lines across 6 new/modified methods)

### Noteworthy Events

Implementation proceeded smoothly with no significant issues:
- Used null coalescing operator (`??`) throughout to ensure backward compatibility
- Applied consistent filtering pattern across all three methods
- Maintained strict type checking with PHP 8.3+ standards
- Cache rebuild confirmed successful integration with ResourcePluginSettingsForm

### Recommendations

**Future Enhancements:**
1. Consider adding a "preview" feature showing which entities would be exposed/hidden with current settings
2. Add configuration schema definition in `config/schema/mcp_server.schema.yml` for proper config validation
3. Document the new settings in user-facing documentation

**Testing:**
- Manual testing recommended at `/admin/config/services/mcp-server/resource-plugins`
- Verify deny list functionality by adding/removing entity types and checking MCP resource list
- Test toggling the canonical URL requirement with entity types that lack canonical templates
