---
id: 1
group: "plugin-configuration"
dependencies: []
status: "completed"
created: 2025-11-20
skills:
  - drupal-plugin-system
  - drupal-form-api
---
# Implement Configuration Form Methods

## Objective

Override configuration methods in ContentEntityResourceTemplate to provide form elements for the "Require Canonical URL" checkbox and entity type deny list, with proper default values and form submission handling.

## Skills Required

- **Drupal Plugin System**: Understanding of plugin configuration architecture, `defaultConfiguration()`, and configuration merging
- **Drupal Form API**: Building form elements (checkbox, checkboxes fieldset), form state handling, and value processing

## Acceptance Criteria

- [ ] Override `defaultConfiguration()` method to return `require_canonical_url` (TRUE) and `denied_entity_types` (empty array)
- [ ] Override `buildConfigurationForm()` method to add checkbox for URL requirement and checkboxes fieldset for entity type deny list
- [ ] Override `submitConfigurationForm()` method to process form values into correct configuration array format
- [ ] Form renders correctly in ResourcePluginSettingsForm at `/admin/config/services/mcp-server/resource-plugins`
- [ ] Configuration saves properly in `mcp_server.resource_plugins` config

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**File to modify**: `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php`

**Methods to add**:

1. `defaultConfiguration()`: Return array with `require_canonical_url` => TRUE and `denied_entity_types` => []
2. `buildConfigurationForm()`:
   - Call `parent::buildConfigurationForm()` first
   - Add single checkbox `$form['require_canonical_url']`
   - Query all content entity types from `$this->entityTypeManager->getDefinitions()`
   - Filter to only content entities implementing `ContentEntityInterface`
   - Build `$form['denied_entity_types']` as checkboxes element with entity type labels
3. `submitConfigurationForm()`:
   - Extract values from form state
   - Use `array_filter()` to remove unchecked items from deny list
   - Ensure `array_values()` to reindex deny list array

**Form structure**:
```php
$form['require_canonical_url'] = [
  '#type' => 'checkbox',
  '#title' => $this->t('Require entities to have canonical URL'),
  '#default_value' => $this->configuration['require_canonical_url'],
  '#description' => $this->t('When checked, only entities with canonical link templates will be exposed as resources.'),
];

$form['denied_entity_types'] = [
  '#type' => 'checkboxes',
  '#title' => $this->t('Denied entity types'),
  '#options' => $entity_type_options,  // ['node' => 'Content', 'user' => 'User', ...]
  '#default_value' => $this->configuration['denied_entity_types'],
  '#description' => $this->t('Select entity types to exclude from MCP resource exposure.'),
];
```

## Input Dependencies

None - this task has no dependencies on other tasks.

## Output Artifacts

- Modified `ContentEntityResourceTemplate.php` with three new methods
- Configuration properly stored in config system

## Implementation Notes

<details>
<summary>Detailed implementation guidance</summary>

### Step 1: Add defaultConfiguration()

Add this method after the `create()` method (around line 91):

```php
/**
 * {@inheritdoc}
 */
protected function defaultConfiguration(): array {
  return parent::defaultConfiguration() + [
    'require_canonical_url' => TRUE,
    'denied_entity_types' => [],
  ];
}
```

### Step 2: Add buildConfigurationForm()

Add this method after `defaultConfiguration()`:

```php
/**
 * {@inheritdoc}
 */
public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
  $form = parent::buildConfigurationForm($form, $form_state);

  $form['require_canonical_url'] = [
    '#type' => 'checkbox',
    '#title' => $this->t('Require entities to have canonical URL'),
    '#default_value' => $this->configuration['require_canonical_url'] ?? TRUE,
    '#description' => $this->t('When checked, only entities with canonical link templates will be exposed as resources.'),
  ];

  // Build entity type options
  $entity_type_options = [];
  foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
    if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
      $entity_type_options[$entity_type_id] = $entity_type->getLabel();
    }
  }

  $form['denied_entity_types'] = [
    '#type' => 'checkboxes',
    '#title' => $this->t('Denied entity types'),
    '#options' => $entity_type_options,
    '#default_value' => $this->configuration['denied_entity_types'] ?? [],
    '#description' => $this->t('Select entity types to exclude from MCP resource exposure.'),
  ];

  return $form;
}
```

### Step 3: Add submitConfigurationForm()

Add this method after `buildConfigurationForm()`:

```php
/**
 * {@inheritdoc}
 */
public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
  parent::submitConfigurationForm($form, $form_state);

  $this->configuration['require_canonical_url'] = (bool) $form_state->getValue('require_canonical_url');

  // Filter out unchecked items (FALSE values) and reindex
  $denied = array_filter($form_state->getValue('denied_entity_types', []));
  $this->configuration['denied_entity_types'] = array_values($denied);
}
```

### Testing Steps

1. Clear cache: `cd /var/www/html && vendor/bin/drush cache:rebuild`
2. Visit `/admin/config/services/mcp-server/resource-plugins`
3. Verify both form elements appear under the Content Entity Resources plugin section
4. Check/uncheck options and save form
5. Verify configuration is saved in config system

</details>
