---
id: 2
group: "plugin-configuration"
dependencies: [1]
status: "completed"
created: 2025-11-20
skills:
  - php-8-development
  - drupal-plugin-system
---
# Update Filtering Logic to Use Configuration

## Objective

Modify the entity filtering logic in `getResources()`, `getResourceContent()`, and `checkAccess()` methods to respect the new configuration options instead of hardcoded canonical URL checks.

## Skills Required

- **PHP 8 Development**: Array operations, conditional logic, and maintaining code quality with strict types
- **Drupal Plugin System**: Understanding plugin configuration access and entity type validation

## Acceptance Criteria

- [ ] `getResources()` method checks `$this->configuration['require_canonical_url']` before applying `hasLinkTemplate('canonical')` filter
- [ ] `getResources()` method excludes entity types present in `$this->configuration['denied_entity_types']`
- [ ] `getResourceContent()` method applies the same filtering logic for consistency
- [ ] `checkAccess()` method applies the same filtering logic for consistency
- [ ] All existing kernel tests pass without modification
- [ ] PHPStan analysis passes with no new errors

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

**File to modify**: `src/Plugin/ResourceTemplate/ContentEntityResourceTemplate.php`

**Methods to update**:

1. **`getResources()`** (lines 103-135):
   - Replace hardcoded `if (!$entity_type->hasLinkTemplate('canonical'))` with conditional check
   - Add deny list check before adding resource to array

2. **`getResourceContent()`** (lines 140-211):
   - Replace hardcoded `if (!$definition->hasLinkTemplate('canonical'))` with conditional check
   - Add deny list check after entity type validation

3. **`checkAccess()`** (lines 216-252):
   - Replace hardcoded `if (!$definition->hasLinkTemplate('canonical'))` with conditional check
   - Add deny list check after entity type validation

**Filtering logic pattern**:

```php
// Check canonical URL requirement if configured
if ($this->configuration['require_canonical_url'] ?? TRUE) {
  if (!$entity_type->hasLinkTemplate('canonical')) {
    continue; // or return NULL depending on method
  }
}

// Check deny list
if (in_array($entity_type_id, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
  continue; // or return NULL depending on method
}
```

## Input Dependencies

- Task 1 must be completed first (configuration methods must exist)
- Configuration array structure: `['require_canonical_url' => bool, 'denied_entity_types' => array]`

## Output Artifacts

- Modified `ContentEntityResourceTemplate.php` with updated filtering logic in 3 methods
- All tests passing
- PHPStan analysis clean

## Implementation Notes

<details>
<summary>Detailed implementation guidance</summary>

### Step 1: Update getResources() method

Locate the loop starting around line 109 and replace the hardcoded canonical URL check:

**Before:**
```php
foreach ($entity_types as $entity_type_id => $entity_type) {
  // Must be content entity with canonical link template.
  if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
    continue;
  }

  if (!$entity_type->hasLinkTemplate('canonical')) {
    continue;
  }
```

**After:**
```php
foreach ($entity_types as $entity_type_id => $entity_type) {
  // Must be content entity with canonical link template.
  if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
    continue;
  }

  // Check canonical URL requirement if configured
  if ($this->configuration['require_canonical_url'] ?? TRUE) {
    if (!$entity_type->hasLinkTemplate('canonical')) {
      continue;
    }
  }

  // Check deny list
  if (in_array($entity_type_id, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
    continue;
  }
```

### Step 2: Update getResourceContent() method

Locate the validation section around line 149-159 and update:

**Before:**
```php
// Validate entity type exists and has canonical link template.
try {
  $definition = $this->entityTypeManager->getDefinition($entity_type);
}
catch (\Exception $e) {
  return NULL;
}

if (!$definition->hasLinkTemplate('canonical')) {
  return NULL;
}
```

**After:**
```php
// Validate entity type exists and has canonical link template.
try {
  $definition = $this->entityTypeManager->getDefinition($entity_type);
}
catch (\Exception $e) {
  return NULL;
}

// Check canonical URL requirement if configured
if ($this->configuration['require_canonical_url'] ?? TRUE) {
  if (!$definition->hasLinkTemplate('canonical')) {
    return NULL;
  }
}

// Check deny list
if (in_array($entity_type, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
  return NULL;
}
```

### Step 3: Update checkAccess() method

Locate the validation section around line 225-235 and update:

**Before:**
```php
// Validate entity type exists and has canonical link template.
try {
  $definition = $this->entityTypeManager->getDefinition($entity_type);
}
catch (\Exception $e) {
  return AccessResult::forbidden('Unknown entity type');
}

if (!$definition->hasLinkTemplate('canonical')) {
  return AccessResult::forbidden('Entity type does not support canonical URLs');
}
```

**After:**
```php
// Validate entity type exists and has canonical link template.
try {
  $definition = $this->entityTypeManager->getDefinition($entity_type);
}
catch (\Exception $e) {
  return AccessResult::forbidden('Unknown entity type');
}

// Check deny list
if (in_array($entity_type, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
  return AccessResult::forbidden('Entity type is denied');
}

// Check canonical URL requirement if configured
if ($this->configuration['require_canonical_url'] ?? TRUE) {
  if (!$definition->hasLinkTemplate('canonical')) {
    return AccessResult::forbidden('Entity type does not support canonical URLs');
  }
}
```

### Step 4: Run tests and static analysis

```bash
cd /var/www/html

# Run existing tests to ensure nothing breaks
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/

# Run PHPStan
vendor/bin/phpstan analyse web/modules/contrib/mcp_server/

# Clear cache after changes
vendor/bin/drush cache:rebuild
```

### Testing Steps

1. Configure plugin with deny list including "user" entity type
2. Verify user entities don't appear in MCP resources list
3. Uncheck "Require Canonical URL" checkbox
4. Verify entities without canonical URLs now appear (if any exist)
5. Re-check checkbox and verify they disappear again

</details>
