---
id: 1
group: "session-management"
dependencies: []
status: "completed"
created: "2025-11-24"
skills:
  - "drupal-backend"
  - "jwt"
---
# Implement JWT Session Manager

## Objective
Replace the current file-based session store with a JWT-based stateless session manager that uses `simple_oauth` signing keys and carries capabilities in verifiable tokens.

## Skills Required
- **drupal-backend**: Service implementation, dependency injection, configuration management
- **jwt**: Token generation, validation, claims structure, signature verification

## Acceptance Criteria
- [ ] `JwtSessionManager` implements `Mcp\Server\Session\SessionInterface`
- [ ] JWT tokens are generated during `initialize` using `simple_oauth` key pairs
- [ ] Tokens contain capability claims (`sampling`, `elicitation`, allowed roots)
- [ ] Inbound tokens are validated (signature, expiry, audience) via `simple_oauth` ResourceServerInterface
- [ ] Invalid/expired tokens return appropriate JSON-RPC errors (401/403)
- [ ] Service is registered as `mcp_server.jwt_session_manager` in `mcp_server.services.yml`
- [ ] Configuration exposes `session.jwt_ttl` and `session.issuer` (defaults: 24h, site URL)
- [ ] `McpServerFactory::create()` uses the JWT manager instead of `FileSessionStore`

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Inject `\Drupal\simple_oauth\KeyRepositoryInterface` for key access
- Inject `\Drupal\simple_oauth\ResourceServerInterface` for validation
- Use `firebase/php-jwt` (provided by `simple_oauth`)
- Create lightweight `SessionContext` value object with capabilities, jti, and expiry
- Configuration keys: `mcp_server.settings:session.jwt_ttl`, `mcp_server.settings:session.issuer`
- Error codes: `401` for missing/invalid token, `403` for insufficient capabilities

## Input Dependencies
None - this is foundational infrastructure

## Output Artifacts
- `src/Session/JwtSessionManager.php` implementing `SessionInterface`
- `src/Session/SessionContext.php` value object
- Service definition in `mcp_server.services.yml`
- Configuration schema updates in `config/schema/mcp_server.schema.yml`
- Updated `McpServerFactory` to use JWT manager

<details>
<summary>Implementation Notes</summary>

### JWT Claims Structure
```php
[
  'iss' => $issuer,        // from config, default site URL
  'aud' => 'mcp-server',   // fixed audience
  'jti' => $uuid,          // unique token ID
  'iat' => time(),         // issued at
  'exp' => time() + $ttl,  // expiry
  'capabilities' => [
    'sampling' => true/false,
    'elicitation' => true/false,
  ],
  'roots' => [...],        // allowed file system roots
]
```

### Session Manager Interface Implementation
- `initialize()`: Generate JWT with capabilities, return to client
- `getSession()`: Validate inbound JWT header, return SessionContext or throw exception
- `close()`: No-op (stateless tokens)

### Error Handling Pattern
```php
try {
  $token = $this->extractTokenFromHeader($request);
  $decoded = $this->resourceServer->validateAccessToken($token);
  return SessionContext::fromToken($decoded);
} catch (\Exception $e) {
  throw new SessionValidationException('Invalid token', 401);
}
```

### Configuration Schema
```yaml
mcp_server.settings:
  type: config_object
  mapping:
    session:
      type: mapping
      mapping:
        jwt_ttl:
          type: integer
          label: 'JWT session TTL in seconds'
        issuer:
          type: string
          label: 'JWT issuer'
```

### Factory Update
Replace:
```php
$session = new FileSessionStore($path);
```
With:
```php
$session = $container->get('mcp_server.jwt_session_manager');
```

### Testing Considerations
- Unit test: Token generation includes all required claims
- Unit test: Validation rejects expired/malformed tokens
- Unit test: Capability checks work correctly
- Kernel test: Service wiring and configuration loading
- Document that revocation is not implemented (future enhancement)

</details>
