---
id: 2
group: "database-coordination"
dependencies: []
status: "completed"
created: "2025-11-24"
skills:
  - "drupal-backend"
  - "database"
---
# Implement Pending Request Repository

## Objective
Create database schema and repository service to coordinate async request/response cycles across PHP-FPM workers.

## Skills Required
- **drupal-backend**: Database API, schema definitions, service implementation
- **database**: Schema design, indexing strategy, transaction handling, status state machines

## Acceptance Criteria
- [ ] `mcp_pending_request` table defined via `hook_schema()`
- [ ] Table includes all required columns: id, correlation_id, session_jti, request_type, request_payload, response_payload, status, expires_at
- [ ] Appropriate indexes created: correlation_id (unique), session_status composite, expires_at
- [ ] `PendingRequestRepository` service implements all required methods
- [ ] Methods handle transaction isolation correctly (SELECT FOR UPDATE)
- [ ] Payloads serialized as JSON for cross-database portability
- [ ] Service registered as `mcp_server.pending_request_repository`
- [ ] Configuration exposes poll_interval_ms, sampling_timeout, elicitation_timeout

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Use Drupal's `hook_schema()` for table definition
- Column types: serial, varchar(64), blob (big), varchar(16), int unsigned
- Indexes: unique on `correlation_id`, composite on `(session_jti, status)`, single on `expires_at`
- Repository methods:
  - `createPending(RequestEnvelope $envelope): PendingRequestId`
  - `storeResponse(PendingRequestId $id, ResponseEnvelope $response): void`
  - `awaitResponse(PendingRequestId $id, \DateTimeImmutable $deadline): ?ResponseEnvelope`
  - `expireOverdue(): int`
- Polling with configurable interval (default 300ms)
- Transaction handling for atomic status updates

## Input Dependencies
None - foundational database infrastructure

## Output Artifacts
- Database schema in `mcp_server.install`
- `src/PendingRequest/PendingRequestRepository.php`
- Value objects: `RequestEnvelope`, `ResponseEnvelope`, `PendingRequestId`
- Service definition in `mcp_server.services.yml`
- Configuration schema for timeouts and polling

<details>
<summary>Implementation Notes</summary>

### Schema Definition
```php
function mcp_server_schema() {
  $schema['mcp_pending_request'] = [
    'description' => 'Stores pending MCP requests awaiting client responses',
    'fields' => [
      'id' => [
        'type' => 'serial',
        'not null' => TRUE,
        'description' => 'Primary Key: Unique request ID',
      ],
      'correlation_id' => [
        'type' => 'varchar',
        'length' => 64,
        'not null' => TRUE,
        'description' => 'UUID correlating request and response',
      ],
      'session_jti' => [
        'type' => 'varchar',
        'length' => 64,
        'not null' => TRUE,
        'description' => 'JWT ID from session token for isolation',
      ],
      'request_type' => [
        'type' => 'varchar',
        'length' => 16,
        'not null' => TRUE,
        'description' => 'Type: sampling or elicitation',
      ],
      'request_payload' => [
        'type' => 'blob',
        'size' => 'big',
        'not null' => TRUE,
        'description' => 'JSON-encoded request parameters',
      ],
      'response_payload' => [
        'type' => 'blob',
        'size' => 'big',
        'not null' => FALSE,
        'description' => 'JSON-encoded response data',
      ],
      'status' => [
        'type' => 'varchar',
        'length' => 16,
        'not null' => TRUE,
        'default' => 'pending',
        'description' => 'Status: pending, completed, expired',
      ],
      'expires_at' => [
        'type' => 'int',
        'unsigned' => TRUE,
        'not null' => TRUE,
        'description' => 'Unix timestamp when request expires',
      ],
    ],
    'primary key' => ['id'],
    'unique keys' => [
      'correlation_id' => ['correlation_id'],
    ],
    'indexes' => [
      'session_status' => ['session_jti', 'status'],
      'expires_at' => ['expires_at'],
    ],
  ];
  return $schema;
}
```

### Repository Implementation Pattern
```php
public function createPending(RequestEnvelope $envelope): PendingRequestId {
  $id = $this->connection->insert('mcp_pending_request')
    ->fields([
      'correlation_id' => $envelope->correlationId,
      'session_jti' => $envelope->sessionJti,
      'request_type' => $envelope->type,
      'request_payload' => json_encode($envelope->payload),
      'status' => 'pending',
      'expires_at' => $envelope->expiresAt->getTimestamp(),
    ])
    ->execute();
  return new PendingRequestId($id);
}

public function storeResponse(PendingRequestId $id, ResponseEnvelope $response): void {
  $txn = $this->connection->startTransaction();
  try {
    // Lock row for update
    $request = $this->connection->select('mcp_pending_request', 'r')
      ->fields('r')
      ->condition('id', $id->value())
      ->condition('status', 'pending')
      ->forUpdate()
      ->execute()
      ->fetchAssoc();

    if (!$request) {
      throw new \RuntimeException('Request not found or already completed');
    }

    $this->connection->update('mcp_pending_request')
      ->fields([
        'response_payload' => json_encode($response->payload),
        'status' => 'completed',
      ])
      ->condition('id', $id->value())
      ->execute();
  } catch (\Exception $e) {
    $txn->rollBack();
    throw $e;
  }
}

public function awaitResponse(PendingRequestId $id, \DateTimeImmutable $deadline): ?ResponseEnvelope {
  $pollInterval = $this->config->get('pending.poll_interval_ms') ?? 300;
  $pollIntervalUs = $pollInterval * 1000;

  while (new \DateTimeImmutable() < $deadline) {
    $row = $this->connection->select('mcp_pending_request', 'r')
      ->fields('r', ['response_payload', 'status'])
      ->condition('id', $id->value())
      ->execute()
      ->fetchAssoc();

    if ($row['status'] === 'completed' && $row['response_payload']) {
      $payload = json_decode($row['response_payload'], true);
      return new ResponseEnvelope($payload);
    }

    if ($row['status'] === 'expired') {
      return null;
    }

    // Check if connection still alive
    if (connection_aborted()) {
      return null;
    }

    usleep($pollIntervalUs);
  }

  // Timeout - mark as expired
  $this->connection->update('mcp_pending_request')
    ->fields(['status' => 'expired'])
    ->condition('id', $id->value())
    ->execute();

  return null;
}

public function expireOverdue(): int {
  return $this->connection->update('mcp_pending_request')
    ->fields(['status' => 'expired'])
    ->condition('status', 'pending')
    ->condition('expires_at', time(), '<')
    ->execute();
}
```

### Configuration Schema
```yaml
mcp_server.settings:
  mapping:
    pending:
      type: mapping
      mapping:
        poll_interval_ms:
          type: integer
          label: 'Poll interval in milliseconds'
        sampling_timeout:
          type: integer
          label: 'Sampling request timeout in seconds'
        elicitation_timeout:
          type: integer
          label: 'Elicitation request timeout in seconds'
```

### Value Objects
```php
final readonly class RequestEnvelope {
  public function __construct(
    public string $correlationId,
    public string $sessionJti,
    public string $type,
    public array $payload,
    public \DateTimeImmutable $expiresAt,
  ) {}
}

final readonly class ResponseEnvelope {
  public function __construct(
    public array $payload,
  ) {}
}

final readonly class PendingRequestId {
  public function __construct(
    private int $id,
  ) {}

  public function value(): int {
    return $this->id;
  }
}
```

### Testing Strategy
- **Unit tests**: Value object construction and serialization
- **Kernel tests**:
  - Schema installation and indexes
  - CRUD operations work correctly
  - Transaction isolation (concurrent storeResponse calls)
  - Poll timeout behavior
  - Expire overdue cleanup
- **Cross-database**: Test with MySQL, MariaDB, PostgreSQL if possible
- Test connection_aborted() behavior in poll loop

</details>
