---
id: 3
group: "request-coordinators"
dependencies: [1, 2]
status: "completed"
created: "2025-11-24"
skills:
  - "drupal-backend"
  - "mcp-protocol"
---
# Implement Sampling Coordinator

## Objective
Provide a single entry point for LLM sampling requests that manages the pending-request lifecycle and returns MCP streaming responses.

## Skills Required
- **drupal-backend**: Service implementation, dependency injection, configuration handling
- **mcp-protocol**: Understanding of sampling request/response format, streaming results, JSON-RPC error handling

## Acceptance Criteria
- [ ] `SamplingCoordinator` service implements `requestCompletion()` method
- [ ] Validates session has `sampling` capability before creating request
- [ ] Generates correlation UUID and creates pending request
- [ ] Returns MCP `StreamingResult` with sampling metadata
- [ ] Polls repository until timeout (configurable, default 25s)
- [ ] Transforms stored response into domain object
- [ ] Returns JSON-RPC error for missing capability or timeout
- [ ] Service registered as `mcp_server.sampling_coordinator`

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Inject `mcp_server.pending_request_repository`
- Inject `@config.factory` for timeout configuration
- Method signature: `requestCompletion(SamplingInstruction $instruction, SessionContext $session): StreamingResult`
- Capability check: `$session->hasCapability('sampling')`
- Generate UUID: `\Ramsey\Uuid\Uuid::uuid4()->toString()`
- Timeout from config: `mcp_server.settings:pending.sampling_timeout` (default 25)
- Return MCP streaming result type for SSE emission

## Input Dependencies
- Task 1: `SessionContext` value object with capabilities
- Task 2: `PendingRequestRepository` for request lifecycle

## Output Artifacts
- `src/Sampling/SamplingCoordinator.php`
- `src/Sampling/SamplingInstruction.php` value object
- Service definition in `mcp_server.services.yml`
- Updated configuration for sampling timeout

<details>
<summary>Implementation Notes</summary>

### Service Implementation
```php
declare(strict_types=1);

namespace Drupal\mcp_server\Sampling;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\mcp_server\PendingRequest\PendingRequestRepository;
use Drupal\mcp_server\PendingRequest\RequestEnvelope;
use Drupal\mcp_server\Session\SessionContext;
use Mcp\Types\StreamingResult;
use Ramsey\Uuid\Uuid;

final readonly class SamplingCoordinator {

  public function __construct(
    private PendingRequestRepository $repository,
    private ConfigFactoryInterface $configFactory,
  ) {}

  public function requestCompletion(
    SamplingInstruction $instruction,
    SessionContext $session
  ): StreamingResult {
    // Validate capability
    if (!$session->hasCapability('sampling')) {
      throw new \RuntimeException('Session does not support sampling capability', 403);
    }

    // Generate correlation ID
    $correlationId = Uuid::uuid4()->toString();

    // Calculate expiry
    $timeout = $this->configFactory
      ->get('mcp_server.settings')
      ->get('pending.sampling_timeout') ?? 25;
    $expiresAt = new \DateTimeImmutable("+{$timeout} seconds");

    // Create pending request
    $envelope = new RequestEnvelope(
      correlationId: $correlationId,
      sessionJti: $session->jti(),
      type: 'sampling',
      payload: $instruction->toArray(),
      expiresAt: $expiresAt,
    );

    $requestId = $this->repository->createPending($envelope);

    // Return streaming result with metadata
    // The transport layer will convert this to SSE
    return new StreamingResult(
      method: 'sampling/createMessage',
      params: [
        'correlationId' => $correlationId,
        'messages' => $instruction->messages,
        'modelPreferences' => $instruction->modelPreferences,
        'systemPrompt' => $instruction->systemPrompt,
        'maxTokens' => $instruction->maxTokens,
      ],
      metadata: [
        'requestId' => $requestId->value(),
        'expiresAt' => $expiresAt->format(\DateTimeInterface::RFC3339),
      ],
    );
  }

  /**
   * Wait for and retrieve the sampling response.
   * Called after SSE emission to resume tool execution.
   */
  public function awaitCompletion(
    PendingRequestId $requestId,
    \DateTimeImmutable $deadline
  ): ?SamplingResponse {
    $responseEnvelope = $this->repository->awaitResponse($requestId, $deadline);

    if ($responseEnvelope === null) {
      return null; // Timeout
    }

    return SamplingResponse::fromArray($responseEnvelope->payload);
  }
}
```

### Value Objects
```php
final readonly class SamplingInstruction {
  public function __construct(
    public array $messages,
    public ?array $modelPreferences = null,
    public ?string $systemPrompt = null,
    public ?int $maxTokens = null,
  ) {}

  public function toArray(): array {
    return array_filter([
      'messages' => $this->messages,
      'modelPreferences' => $this->modelPreferences,
      'systemPrompt' => $this->systemPrompt,
      'maxTokens' => $this->maxTokens,
    ], fn($v) => $v !== null);
  }
}

final readonly class SamplingResponse {
  public function __construct(
    public string $role,
    public string $content,
    public ?string $model = null,
    public ?string $stopReason = null,
  ) {}

  public static function fromArray(array $data): self {
    return new self(
      role: $data['role'] ?? 'assistant',
      content: $data['content']['text'] ?? '',
      model: $data['model'] ?? null,
      stopReason: $data['stopReason'] ?? null,
    );
  }
}
```

### Integration with Tool Handlers
The coordinator is injected into `CustomCallToolHandler`:

```php
// In CustomCallToolHandler
public function __construct(
  // ... existing dependencies
  private readonly SamplingCoordinator $samplingCoordinator,
) {}

// When tool requests sampling:
$instruction = new SamplingInstruction(
  messages: [['role' => 'user', 'content' => ['type' => 'text', 'text' => 'Analyze this data']]],
  maxTokens: 1000,
);

$streamingResult = $this->samplingCoordinator->requestCompletion(
  $instruction,
  $this->sessionContext
);

// Return the streaming result to propagate up to transport layer
return $streamingResult;
```

### Error Handling
```php
// Missing capability
if (!$session->hasCapability('sampling')) {
  throw new JsonRpcException(
    'Sampling capability not enabled for this session',
    -32001, // Custom error code
    ['capability' => 'sampling']
  );
}

// Timeout
$response = $this->awaitCompletion($requestId, $deadline);
if ($response === null) {
  throw new JsonRpcException(
    'Sampling request timed out',
    -32002,
    ['requestId' => $requestId->value(), 'timeout' => $timeout]
  );
}
```

### Configuration Addition
```yaml
# config/schema/mcp_server.schema.yml
mcp_server.settings:
  mapping:
    pending:
      mapping:
        sampling_timeout:
          type: integer
          label: 'Sampling request timeout in seconds'
          default: 25
```

### Testing Considerations
- **Unit tests**:
  - Capability validation (reject missing capability)
  - Request envelope construction
  - Correlation ID generation (UUID v4 format)
  - Timeout calculation
- **Kernel tests**:
  - End-to-end flow: create request, store response, retrieve
  - Timeout behavior (mock clock or short timeout)
  - Service wiring and dependency injection
- **Mock the repository** in unit tests to avoid database dependency
- Test serialization/deserialization of SamplingInstruction and SamplingResponse

</details>
