---
id: 4
group: "request-coordinators"
dependencies: [1, 2]
status: "completed"
created: "2025-11-24"
skills:
  - "drupal-backend"
  - "mcp-protocol"
---
# Implement Elicitation Coordinator

## Objective
Provide a single entry point for user input requests that manages the pending-request lifecycle with longer timeouts and UI metadata.

## Skills Required
- **drupal-backend**: Service implementation, dependency injection, configuration handling
- **mcp-protocol**: Understanding of elicitation request/response format, prompt types (text, select, multi-select), streaming results

## Acceptance Criteria
- [ ] `ElicitationCoordinator` service implements `requestInput()` method
- [ ] Validates session has `elicitation` capability before creating request
- [ ] Supports prompt types: text, select, multi-select
- [ ] Generates correlation UUID and creates pending request
- [ ] Returns MCP `StreamingResult` with elicitation metadata
- [ ] Polls repository until timeout (configurable, default 60s)
- [ ] Normalizes user responses into value object
- [ ] Validates response format and returns JSON-RPC errors for invalid data
- [ ] Service registered as `mcp_server.elicitation_coordinator`

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Inject `mcp_server.pending_request_repository`
- Inject `@config.factory` for timeout configuration
- Method signature: `requestInput(ElicitationPrompt $prompt, SessionContext $session): StreamingResult`
- Capability check: `$session->hasCapability('elicitation')`
- Generate UUID: `\Ramsey\Uuid\Uuid::uuid4()->toString()`
- Timeout from config: `mcp_server.settings:pending.elicitation_timeout` (default 60)
- Support prompt types: `text`, `select`, `multi-select`

## Input Dependencies
- Task 1: `SessionContext` value object with capabilities
- Task 2: `PendingRequestRepository` for request lifecycle

## Output Artifacts
- `src/Elicitation/ElicitationCoordinator.php`
- `src/Elicitation/ElicitationPrompt.php` value object
- `src/Elicitation/ElicitationResponse.php` value object
- Service definition in `mcp_server.services.yml`
- Updated configuration for elicitation timeout

<details>
<summary>Implementation Notes</summary>

### Service Implementation
```php
declare(strict_types=1);

namespace Drupal\mcp_server\Elicitation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\mcp_server\PendingRequest\PendingRequestRepository;
use Drupal\mcp_server\PendingRequest\RequestEnvelope;
use Drupal\mcp_server\PendingRequest\PendingRequestId;
use Drupal\mcp_server\Session\SessionContext;
use Mcp\Types\StreamingResult;
use Ramsey\Uuid\Uuid;

final readonly class ElicitationCoordinator {

  public function __construct(
    private PendingRequestRepository $repository,
    private ConfigFactoryInterface $configFactory,
  ) {}

  public function requestInput(
    ElicitationPrompt $prompt,
    SessionContext $session
  ): StreamingResult {
    // Validate capability
    if (!$session->hasCapability('elicitation')) {
      throw new \RuntimeException('Session does not support elicitation capability', 403);
    }

    // Generate correlation ID
    $correlationId = Uuid::uuid4()->toString();

    // Calculate expiry (longer timeout for user interaction)
    $timeout = $this->configFactory
      ->get('mcp_server.settings')
      ->get('pending.elicitation_timeout') ?? 60;
    $expiresAt = new \DateTimeImmutable("+{$timeout} seconds");

    // Create pending request
    $envelope = new RequestEnvelope(
      correlationId: $correlationId,
      sessionJti: $session->jti(),
      type: 'elicitation',
      payload: $prompt->toArray(),
      expiresAt: $expiresAt,
    );

    $requestId = $this->repository->createPending($envelope);

    // Return streaming result with metadata
    return new StreamingResult(
      method: 'prompts/get',
      params: [
        'correlationId' => $correlationId,
        'prompt' => $prompt->toArray(),
      ],
      metadata: [
        'requestId' => $requestId->value(),
        'expiresAt' => $expiresAt->format(\DateTimeInterface::RFC3339),
      ],
    );
  }

  /**
   * Wait for and retrieve the user response.
   */
  public function awaitInput(
    PendingRequestId $requestId,
    \DateTimeImmutable $deadline
  ): ?ElicitationResponse {
    $responseEnvelope = $this->repository->awaitResponse($requestId, $deadline);

    if ($responseEnvelope === null) {
      return null; // Timeout
    }

    return ElicitationResponse::fromArray($responseEnvelope->payload);
  }
}
```

### Value Objects
```php
final readonly class ElicitationPrompt {
  public const TYPE_TEXT = 'text';
  public const TYPE_SELECT = 'select';
  public const TYPE_MULTI_SELECT = 'multi-select';

  public function __construct(
    public string $type,
    public string $question,
    public ?array $options = null,
    public ?string $defaultValue = null,
    public ?string $placeholder = null,
  ) {
    if (!in_array($type, [self::TYPE_TEXT, self::TYPE_SELECT, self::TYPE_MULTI_SELECT], true)) {
      throw new \InvalidArgumentException("Invalid prompt type: {$type}");
    }

    if (in_array($type, [self::TYPE_SELECT, self::TYPE_MULTI_SELECT], true) && empty($options)) {
      throw new \InvalidArgumentException("Options required for {$type} prompt");
    }
  }

  public function toArray(): array {
    return array_filter([
      'type' => $this->type,
      'question' => $this->question,
      'options' => $this->options,
      'defaultValue' => $this->defaultValue,
      'placeholder' => $this->placeholder,
    ], fn($v) => $v !== null);
  }

  public static function text(
    string $question,
    ?string $defaultValue = null,
    ?string $placeholder = null
  ): self {
    return new self(self::TYPE_TEXT, $question, null, $defaultValue, $placeholder);
  }

  public static function select(
    string $question,
    array $options,
    ?string $defaultValue = null
  ): self {
    return new self(self::TYPE_SELECT, $question, $options, $defaultValue);
  }

  public static function multiSelect(
    string $question,
    array $options,
    ?array $defaultValues = null
  ): self {
    return new self(
      self::TYPE_MULTI_SELECT,
      $question,
      $options,
      $defaultValues ? json_encode($defaultValues) : null
    );
  }
}

final readonly class ElicitationResponse {
  public function __construct(
    public string|array $value,
  ) {}

  public static function fromArray(array $data): self {
    // Handle both single values and arrays (for multi-select)
    $value = $data['value'] ?? $data['values'] ?? '';
    return new self($value);
  }

  public function asString(): string {
    return is_array($this->value) ? implode(', ', $this->value) : $this->value;
  }

  public function asArray(): array {
    return is_array($this->value) ? $this->value : [$this->value];
  }
}
```

### Integration Example
```php
// In a tool handler
$prompt = ElicitationPrompt::select(
  question: 'Which format should I export to?',
  options: [
    ['id' => 'json', 'label' => 'JSON'],
    ['id' => 'csv', 'label' => 'CSV'],
    ['id' => 'xml', 'label' => 'XML'],
  ]
);

$streamingResult = $this->elicitationCoordinator->requestInput(
  $prompt,
  $this->sessionContext
);

// This returns to transport layer, which emits SSE
// Later, after client responds:
$response = $this->elicitationCoordinator->awaitInput($requestId, $deadline);
if ($response === null) {
  throw new \RuntimeException('User input timed out');
}

$format = $response->asString(); // 'json', 'csv', or 'xml'
```

### Response Validation
```php
private function validateResponse(array $responseData, ElicitationPrompt $prompt): void {
  switch ($prompt->type) {
    case ElicitationPrompt::TYPE_TEXT:
      if (!isset($responseData['value']) || !is_string($responseData['value'])) {
        throw new \InvalidArgumentException('Text response must contain string value');
      }
      break;

    case ElicitationPrompt::TYPE_SELECT:
      if (!isset($responseData['value']) || !is_string($responseData['value'])) {
        throw new \InvalidArgumentException('Select response must contain string value');
      }
      // Optionally validate value is in options
      $validIds = array_column($prompt->options ?? [], 'id');
      if (!in_array($responseData['value'], $validIds, true)) {
        throw new \InvalidArgumentException('Invalid option selected');
      }
      break;

    case ElicitationPrompt::TYPE_MULTI_SELECT:
      if (!isset($responseData['values']) || !is_array($responseData['values'])) {
        throw new \InvalidArgumentException('Multi-select response must contain array of values');
      }
      break;
  }
}
```

### Error Handling
```php
// Missing capability
if (!$session->hasCapability('elicitation')) {
  throw new JsonRpcException(
    'Elicitation capability not enabled for this session',
    -32001,
    ['capability' => 'elicitation']
  );
}

// Invalid response format
try {
  $this->validateResponse($responseData, $prompt);
} catch (\InvalidArgumentException $e) {
  throw new JsonRpcException(
    'Invalid elicitation response format',
    -32003,
    ['error' => $e->getMessage()]
  );
}

// Timeout
$response = $this->awaitInput($requestId, $deadline);
if ($response === null) {
  throw new JsonRpcException(
    'Elicitation request timed out',
    -32002,
    ['requestId' => $requestId->value(), 'timeout' => $timeout]
  );
}
```

### Configuration Addition
```yaml
# config/schema/mcp_server.schema.yml
mcp_server.settings:
  mapping:
    pending:
      mapping:
        elicitation_timeout:
          type: integer
          label: 'Elicitation request timeout in seconds'
          default: 60
```

### Testing Considerations
- **Unit tests**:
  - Prompt validation (reject invalid types, missing options)
  - Response normalization (single vs array values)
  - Response validation for each prompt type
  - Named constructors (text, select, multiSelect)
- **Kernel tests**:
  - End-to-end flow for each prompt type
  - Timeout behavior
  - Service wiring
- **Integration tests**:
  - Test with actual MCP Inspector client
  - Verify option rendering and selection
  - Test multi-select array handling

### Shared Utilities with Sampling Coordinator
Consider extracting common polling/timeout logic into a trait:

```php
trait PendingRequestPoller {
  private function pollUntilComplete(
    PendingRequestId $requestId,
    \DateTimeImmutable $deadline
  ): ?ResponseEnvelope {
    return $this->repository->awaitResponse($requestId, $deadline);
  }

  private function calculateDeadline(int $timeoutSeconds): \DateTimeImmutable {
    return new \DateTimeImmutable("+{$timeoutSeconds} seconds");
  }
}
```

</details>
