---
id: 5
group: "transport-layer"
dependencies: [2, 3, 4]
status: "completed"
created: "2025-11-24"
skills:
  - "drupal-backend"
  - "http-streaming"
---
# Implement SSE Transport Adapter

## Objective
Upgrade the MCP controller to detect streaming results from coordinators and emit Server-Sent Events while keeping the PHP process responsive during repository polling.

## Skills Required
- **drupal-backend**: Controller modifications, Symfony responses, service injection
- **http-streaming**: SSE format, keep-alive mechanics, connection monitoring, chunked transfer encoding

## Acceptance Criteria
- [ ] `McpServerController::handle()` detects `StreamingResultInterface` responses
- [ ] Streaming results trigger Symfony `StreamedResponse` with SSE formatting
- [ ] SSE frames follow standard format: `data: {...}\n\n`
- [ ] Keep-alive comments sent every ~3 seconds during polling
- [ ] Connection status monitored via `connection_aborted()`
- [ ] Final result or timeout error emitted as SSE frame before stream closure
- [ ] Non-streaming requests maintain existing JSON response behavior
- [ ] Correlation IDs logged for observability

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Inject `PendingRequestRepository` into controller
- Create `StreamingResponseEmitter` helper service for SSE formatting
- Use Symfony `StreamedResponse` with appropriate headers:
  - `Content-Type: text/event-stream`
  - `Cache-Control: no-cache`
  - `X-Accel-Buffering: no` (for nginx)
- Monitor `connection_aborted()` during polling
- Emit keep-alive every 3 seconds: `: keep-alive\n\n`
- Use `flush()` after each SSE write

## Input Dependencies
- Task 2: `PendingRequestRepository` for polling
- Task 3: `SamplingCoordinator` returns `StreamingResult`
- Task 4: `ElicitationCoordinator` returns `StreamingResult`

## Output Artifacts
- Updated `src/Controller/McpServerController.php`
- `src/Streaming/StreamingResponseEmitter.php` helper
- Service definition for streaming emitter
- Logging integration for correlation IDs

<details>
<summary>Implementation Notes</summary>

### Controller Enhancement
```php
declare(strict_types=1);

namespace Drupal\mcp_server\Controller;

use Drupal\mcp_server\PendingRequest\PendingRequestRepository;
use Drupal\mcp_server\Streaming\StreamingResponseEmitter;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class McpServerController extends ControllerBase {

  public function __construct(
    // ... existing dependencies
    private readonly PendingRequestRepository $repository,
    private readonly StreamingResponseEmitter $emitter,
  ) {
    parent::__construct();
  }

  public static function create(ContainerInterface $container) {
    return new static(
      // ... existing dependencies
      $container->get('mcp_server.pending_request_repository'),
      $container->get('mcp_server.streaming_response_emitter'),
    );
  }

  public function handle(Request $request): Response {
    try {
      // Existing MCP request processing
      $psrRequest = $this->httpMessageFactory->createRequest($request);
      $psrResponse = $this->transport->handle($psrRequest);

      // Check if response contains streaming result
      $body = (string) $psrResponse->getBody();
      $data = json_decode($body, true);

      if ($this->isStreamingResult($data)) {
        return $this->handleStreamingResponse($data);
      }

      // Standard JSON response
      return $this->httpFoundationFactory->createResponse($psrResponse);

    } catch (\Exception $e) {
      $this->logger->error('MCP request failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return new JsonResponse(['error' => $e->getMessage()], 500);
    }
  }

  private function isStreamingResult(array $data): bool {
    return isset($data['result']['_streaming'])
      && isset($data['result']['_metadata']['requestId']);
  }

  private function handleStreamingResponse(array $data): StreamedResponse {
    $metadata = $data['result']['_metadata'];
    $requestId = new PendingRequestId($metadata['requestId']);
    $correlationId = $data['result']['params']['correlationId'] ?? 'unknown';
    $expiresAt = new \DateTimeImmutable($metadata['expiresAt']);

    $this->logger->info('Starting SSE stream for correlation @id', [
      '@id' => $correlationId,
    ]);

    $response = new StreamedResponse();
    $response->headers->set('Content-Type', 'text/event-stream');
    $response->headers->set('Cache-Control', 'no-cache');
    $response->headers->set('X-Accel-Buffering', 'no');

    $response->setCallback(function () use ($requestId, $expiresAt, $correlationId, $data) {
      // Emit initial request
      $this->emitter->emitEvent('request', $data['result']['params']);

      // Poll for response
      $responseEnvelope = $this->repository->awaitResponse($requestId, $expiresAt);

      if ($responseEnvelope === null) {
        // Timeout
        $this->emitter->emitEvent('error', [
          'code' => -32002,
          'message' => 'Request timed out',
          'data' => ['correlationId' => $correlationId],
        ]);
        $this->logger->warning('SSE stream timed out for correlation @id', [
          '@id' => $correlationId,
        ]);
      } else {
        // Success
        $this->emitter->emitEvent('response', $responseEnvelope->payload);
        $this->logger->info('SSE stream completed for correlation @id', [
          '@id' => $correlationId,
        ]);
      }

      // Final event to signal completion
      $this->emitter->emitEvent('done', ['status' => 'completed']);
    });

    return $response;
  }
}
```

### StreamingResponseEmitter
```php
declare(strict_types=1);

namespace Drupal\mcp_server\Streaming;

final class StreamingResponseEmitter {

  private int $lastKeepAlive = 0;

  public function emitEvent(string $event, array $data): void {
    echo "event: {$event}\n";
    echo "data: " . json_encode($data) . "\n\n";
    flush();

    // Update keep-alive timestamp
    $this->lastKeepAlive = time();
  }

  public function emitKeepAlive(): void {
    $now = time();
    if ($now - $this->lastKeepAlive >= 3) {
      echo ": keep-alive\n\n";
      flush();
      $this->lastKeepAlive = $now;
    }
  }

  public function shouldContinue(): bool {
    return !connection_aborted();
  }
}
```

### Enhanced Repository awaitResponse with Keep-Alive
```php
// In PendingRequestRepository
public function awaitResponse(
  PendingRequestId $id,
  \DateTimeImmutable $deadline
): ?ResponseEnvelope {
  $pollInterval = $this->config->get('pending.poll_interval_ms') ?? 300;
  $pollIntervalUs = $pollInterval * 1000;

  while (new \DateTimeImmutable() < $deadline) {
    // Check connection status
    if (connection_aborted()) {
      $this->logger->info('Client disconnected, aborting poll for request @id', [
        '@id' => $id->value(),
      ]);
      return null;
    }

    // Check for response
    $row = $this->connection->select('mcp_pending_request', 'r')
      ->fields('r', ['response_payload', 'status'])
      ->condition('id', $id->value())
      ->execute()
      ->fetchAssoc();

    if ($row['status'] === 'completed' && $row['response_payload']) {
      $payload = json_decode($row['response_payload'], true);
      return new ResponseEnvelope($payload);
    }

    if ($row['status'] === 'expired') {
      return null;
    }

    // Emit keep-alive before sleeping
    // Note: This requires injecting the emitter or using a callback
    usleep($pollIntervalUs);
  }

  // Timeout
  $this->markExpired($id);
  return null;
}

private function markExpired(PendingRequestId $id): void {
  $this->connection->update('mcp_pending_request')
    ->fields(['status' => 'expired'])
    ->condition('id', $id->value())
    ->execute();
}
```

### Alternative: Callback-Based Keep-Alive
```php
public function awaitResponse(
  PendingRequestId $id,
  \DateTimeImmutable $deadline,
  ?callable $keepAliveCallback = null
): ?ResponseEnvelope {
  // ... existing polling logic ...

  while (new \DateTimeImmutable() < $deadline) {
    // Send keep-alive if callback provided
    if ($keepAliveCallback !== null) {
      $keepAliveCallback();
    }

    // ... rest of polling ...
  }
}

// In controller:
$responseEnvelope = $this->repository->awaitResponse(
  $requestId,
  $expiresAt,
  fn() => $this->emitter->emitKeepAlive()
);
```

### SSE Event Format Examples
```
Initial request emission:
event: request
data: {"correlationId":"123e4567-e89b-12d3-a456-426614174000","messages":[...]}

Keep-alive comments:
: keep-alive

Final response:
event: response
data: {"role":"assistant","content":{"type":"text","text":"Analysis complete"},"model":"claude-3"}

Completion signal:
event: done
data: {"status":"completed"}

Error case:
event: error
data: {"code":-32002,"message":"Request timed out","data":{"correlationId":"..."}}
```

### Service Definition
```yaml
# mcp_server.services.yml
services:
  mcp_server.streaming_response_emitter:
    class: Drupal\mcp_server\Streaming\StreamingResponseEmitter
```

### Testing Considerations
- **Unit tests**:
  - `isStreamingResult()` detection logic
  - SSE format validation (event/data structure)
  - Keep-alive timing logic
- **Kernel tests**:
  - Mock streaming result and verify response type
  - Verify headers set correctly
  - Test connection_aborted() handling
- **Functional tests**:
  - Full SSE stream with actual client
  - Test with MCP Inspector
  - Verify keep-alive received during long polling
  - Test early client disconnect
- **Integration tests**:
  - End-to-end: tool triggers sampling → SSE stream → client responds → final result

### Debugging and Observability
```php
// Add debug logging
$this->logger->debug('SSE: Emitting @event event', [
  '@event' => $event,
  '@data' => json_encode($data),
]);

// Monitor timing
$startTime = microtime(true);
$responseEnvelope = $this->repository->awaitResponse($requestId, $expiresAt);
$duration = microtime(true) - $startTime;

$this->logger->info('SSE poll completed in @duration seconds', [
  '@duration' => number_format($duration, 3),
  '@status' => $responseEnvelope ? 'success' : 'timeout',
]);
```

### Known Issues and Mitigations
1. **PHP output buffering**: May interfere with flush()
   - Mitigation: Disable output buffering for SSE routes, or call `ob_flush()` after `flush()`

2. **Nginx buffering**: Can delay SSE delivery
   - Mitigation: Set `X-Accel-Buffering: no` header

3. **PHP-FPM timeouts**: Default 30s may be too short
   - Mitigation: Document recommended PHP-FPM settings (request_terminate_timeout)

4. **Browser/client SSE support**: Varies by client
   - Mitigation: Follow MCP spec exactly; test with MCP Inspector

</details>
