---
id: 6
group: "transport-layer"
dependencies: [1, 2, 5]
status: "completed"
created: "2025-11-24"
completed: "2025-11-24"
skills:
  - "drupal-backend"
  - "mcp-protocol"
---
# Implement Response Routing and Validation

## Objective
Accept asynchronous client responses, validate them against session context, and safely resume waiting tool executions while preventing cross-session injection attacks.

## Skills Required
- **drupal-backend**: Request routing, validation logic, transaction handling
- **mcp-protocol**: JSON-RPC response format, correlation ID handling, error codes

## Acceptance Criteria
- [ ] Controller branches incoming payloads: method-based requests vs responses
- [ ] Responses validated: correlation ID exists, session JTI matches, status is pending
- [ ] Invalid responses return 404/409 with appropriate JSON-RPC errors
- [ ] Valid responses persisted via repository and return HTTP 202
- [ ] Cron task calls `expireOverdue()` for cleanup
- [ ] Expired requests marked for deterministic error messaging
- [ ] Cross-session response injection prevented through JTI validation
- [ ] Correlation IDs logged for security audit trail

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
- Extend `McpServerController::handle()` to detect response payloads
- Response detection: has `result`/`error` but no `method` key
- Validation chain: correlation ID → session JTI → pending status
- Use `PendingRequestRepository::storeResponse()` for persistence
- Implement `hook_cron()` to call repository cleanup
- Security: Prevent timing attacks in validation (constant-time comparison where appropriate)
- Return HTTP 202 Accepted for async acknowledgment

## Input Dependencies
- Task 1: `SessionContext` with JTI for validation
- Task 2: `PendingRequestRepository` for storage and cleanup
- Task 5: SSE transport layer that polls for responses

## Output Artifacts
- Updated `src/Controller/McpServerController.php` with response routing
- `mcp_server_cron()` implementation for cleanup
- Enhanced validation and security logging
- Error response templates for validation failures

<details>
<summary>Implementation Notes</summary>

### Controller Request Branching
```php
public function handle(Request $request): Response {
  try {
    $psrRequest = $this->httpMessageFactory->createRequest($request);
    $body = json_decode($psrRequest->getBody()->getContents(), true);

    // Branch on request type
    if ($this->isResponse($body)) {
      return $this->handleClientResponse($body, $psrRequest);
    }

    // Standard MCP request (tool call, resource access, etc.)
    $psrResponse = $this->transport->handle($psrRequest);

    if ($this->isStreamingResult($psrResponse)) {
      return $this->handleStreamingResponse($psrResponse);
    }

    return $this->httpFoundationFactory->createResponse($psrResponse);

  } catch (\Exception $e) {
    return $this->handleException($e);
  }
}

private function isResponse(array $body): bool {
  // Response has result/error but no method
  return (isset($body['result']) || isset($body['error']))
    && !isset($body['method']);
}
```

### Response Handler Implementation
```php
private function handleClientResponse(array $body, ServerRequestInterface $psrRequest): Response {
  // Extract correlation ID
  $correlationId = $body['id'] ?? null;
  if ($correlationId === null) {
    return $this->jsonRpcError(-32600, 'Missing correlation ID', 400);
  }

  // Validate session and extract JTI
  try {
    $session = $this->sessionManager->getSession($psrRequest);
  } catch (SessionValidationException $e) {
    $this->logger->warning('Response rejected: invalid session for correlation @id', [
      '@id' => $correlationId,
    ]);
    return $this->jsonRpcError(-32001, 'Invalid session', 401);
  }

  // Look up pending request
  try {
    $pendingRequest = $this->repository->findByCorrelationId($correlationId);
  } catch (\RuntimeException $e) {
    $this->logger->warning('Response rejected: unknown correlation @id', [
      '@id' => $correlationId,
    ]);
    return $this->jsonRpcError(-32004, 'Unknown correlation ID', 404);
  }

  // Validate session isolation
  if ($pendingRequest['session_jti'] !== $session->jti()) {
    $this->logger->error('Response injection attempt: JTI mismatch for correlation @id', [
      '@id' => $correlationId,
      '@expected_jti' => $pendingRequest['session_jti'],
      '@actual_jti' => $session->jti(),
    ]);
    return $this->jsonRpcError(-32005, 'Session mismatch', 403);
  }

  // Validate request is still pending
  if ($pendingRequest['status'] !== 'pending') {
    $this->logger->info('Response rejected: request already @status for correlation @id', [
      '@status' => $pendingRequest['status'],
      '@id' => $correlationId,
    ]);
    return $this->jsonRpcError(-32006, "Request already {$pendingRequest['status']}", 409);
  }

  // Store response
  try {
    $requestId = new PendingRequestId($pendingRequest['id']);
    $responseEnvelope = new ResponseEnvelope($body['result'] ?? $body['error']);
    $this->repository->storeResponse($requestId, $responseEnvelope);

    $this->logger->info('Response accepted for correlation @id', [
      '@id' => $correlationId,
    ]);

    return new JsonResponse(
      ['status' => 'accepted', 'correlationId' => $correlationId],
      202
    );
  } catch (\Exception $e) {
    $this->logger->error('Failed to store response for correlation @id: @error', [
      '@id' => $correlationId,
      '@error' => $e->getMessage(),
    ]);
    return $this->jsonRpcError(-32603, 'Internal error storing response', 500);
  }
}
```

### Repository Enhancement for Lookup
```php
// Add to PendingRequestRepository
public function findByCorrelationId(string $correlationId): array {
  $row = $this->connection->select('mcp_pending_request', 'r')
    ->fields('r')
    ->condition('correlation_id', $correlationId)
    ->execute()
    ->fetchAssoc();

  if (!$row) {
    throw new \RuntimeException("No pending request found for correlation ID: {$correlationId}");
  }

  return $row;
}
```

### Error Response Helper
```php
private function jsonRpcError(int $code, string $message, int $httpStatus): JsonResponse {
  return new JsonResponse(
    [
      'jsonrpc' => '2.0',
      'error' => [
        'code' => $code,
        'message' => $message,
      ],
      'id' => null,
    ],
    $httpStatus
  );
}
```

### Cron Implementation
```php
/**
 * Implements hook_cron().
 */
function mcp_server_cron() {
  $repository = \Drupal::service('mcp_server.pending_request_repository');
  $count = $repository->expireOverdue();

  if ($count > 0) {
    \Drupal::logger('mcp_server')->info('Expired @count overdue pending requests', [
      '@count' => $count,
    ]);
  }
}
```

### Enhanced expireOverdue with Monitoring
```php
public function expireOverdue(): int {
  $now = time();

  // Mark expired
  $expired = $this->connection->update('mcp_pending_request')
    ->fields(['status' => 'expired'])
    ->condition('status', 'pending')
    ->condition('expires_at', $now, '<')
    ->execute();

  // Delete old completed/expired records (older than 24 hours)
  $cutoff = $now - 86400;
  $deleted = $this->connection->delete('mcp_pending_request')
    ->condition('status', ['completed', 'expired'], 'IN')
    ->condition('expires_at', $cutoff, '<')
    ->execute();

  if ($deleted > 0) {
    $this->logger->info('Cleaned up @count old pending requests', [
      '@count' => $deleted,
    ]);
  }

  return $expired;
}
```

### Security Considerations

**Timing Attack Prevention**:
```php
// Use hash_equals for constant-time string comparison
if (!hash_equals($pendingRequest['session_jti'], $session->jti())) {
  // ... reject
}
```

**Rate Limiting** (optional future enhancement):
```php
// Track failed validation attempts per IP
// Implement exponential backoff for repeated failures
// Document for operators to configure at reverse proxy level
```

**Audit Logging**:
```php
// Log all validation failures with context
$this->logger->warning('Response validation failed', [
  '@reason' => $reason,
  '@correlation_id' => $correlationId,
  '@session_jti' => $session->jti(),
  '@client_ip' => $request->getClientIp(),
  '@user_agent' => $request->headers->get('User-Agent'),
]);
```

### Error Code Registry
Document all custom JSON-RPC error codes:

```
-32001: Invalid session (401)
-32002: Request timed out (internal, from coordinator)
-32003: Invalid elicitation response format (400)
-32004: Unknown correlation ID (404)
-32005: Session mismatch / injection attempt (403)
-32006: Request already completed/expired (409)
```

### Testing Considerations
- **Unit tests**:
  - Request type detection (isResponse)
  - Validation logic for each failure case
  - Error response formatting
- **Kernel tests**:
  - End-to-end: create request → store response → verify status
  - Cross-session injection attempt (different JTI)
  - Duplicate response handling (already completed)
  - Cron cleanup removes old records
- **Security tests**:
  - Timing attack resistance in JTI comparison
  - Invalid correlation ID handling
  - Expired request handling
- **Integration tests**:
  - Full sampling flow: request → SSE → client response → completion
  - Full elicitation flow: request → SSE → client response → completion
  - Timeout scenario: request → no response → cron marks expired

### Monitoring and Observability
```php
// Add metrics collection hooks for operators
\Drupal::moduleHandler()->invokeAll('mcp_server_response_stored', [
  'correlation_id' => $correlationId,
  'request_type' => $pendingRequest['request_type'],
  'duration_ms' => $durationMs,
]);

\Drupal::moduleHandler()->invokeAll('mcp_server_validation_failed', [
  'reason' => $reason,
  'correlation_id' => $correlationId,
]);
```

### Performance Considerations
- **Index usage**: Ensure `correlation_id` unique index used for lookups
- **Transaction isolation**: Use `FOR UPDATE` only when necessary (storeResponse)
- **Connection pooling**: Monitor database connection usage during high load
- **Cleanup frequency**: Balance between cleanup overhead and table size

</details>
