---
id: 7
group: "integration-testing"
dependencies: [1, 2, 3, 4, 5, 6]
status: "completed"
created: "2025-11-24"
completed: "2025-11-24"
skills:
  - "drupal-backend"
  - "testing"
---
# Integration and Testing

## Objective
Implement comprehensive test coverage across all components and verify end-to-end sampling and elicitation flows work correctly.

## Skills Required
- **drupal-backend**: PHPUnit, kernel tests, functional tests, service mocking
- **testing**: Test strategy, integration testing, mock clients, test data generation

## Acceptance Criteria
- [ ] Unit tests cover JWT manager, coordinators, value objects
- [ ] Kernel tests verify database operations and service wiring
- [ ] Integration tests demonstrate full request/response cycles
- [ ] Test coverage includes timeout scenarios and error conditions
- [ ] Security tests validate session isolation and injection prevention
- [ ] Documentation includes testing guide for MCP Inspector
- [ ] All tests pass with Drupal's PHPUnit configuration
- [ ] Feature flags tested (enabled/disabled states)

Use your internal Todo tool to track these and keep on track.

## Technical Requirements
**CRITICAL - Meaningful Test Strategy**:

Your critical mantra for test generation is: "write a few tests, mostly integration".

**Definition of "Meaningful Tests":**
Tests that verify custom business logic, critical paths, and edge cases specific to this implementation. Focus on testing YOUR code, not the framework or library functionality.

**When TO Write Tests:**
- JWT token generation and validation logic
- Pending request state transitions and coordination
- Coordinator capability validation and timeout handling
- SSE emission and polling behavior
- Cross-session security boundaries
- Request/response correlation and validation

**When NOT to Write Tests:**
- `simple_oauth` JWT signature verification (tested upstream)
- Drupal database API behavior (tested by Drupal)
- `firebase/php-jwt` library functions (tested by library)
- Symfony response object behavior (tested by Symfony)
- Basic getter/setter methods or property access

**Test Organization**:
- Unit tests in `tests/src/Unit/`
- Kernel tests in `tests/src/Kernel/`
- Functional test in `tests/src/Functional/`
- ONE functional test class maximum
- ONE test method per functional test class (use helpers)

## Input Dependencies
All previous tasks (1-6) must be complete

## Output Artifacts
- Unit test suite in `tests/src/Unit/`
- Kernel test suite in `tests/src/Kernel/`
- One functional test in `tests/src/Functional/`
- Test documentation in module README or separate TESTING.md
- Mock client utilities for testing

<details>
<summary>Implementation Notes</summary>

### Meaningful Test Strategy Guidelines

**IMPORTANT**: Keep in mind the _Meaningful Test Strategy Guidelines_ from the task generation instructions.

### Unit Test Coverage

**JWT Session Manager Tests** (`tests/src/Unit/Session/JwtSessionManagerTest.php`):
```php
class JwtSessionManagerTest extends UnitTestCase {

  public function testTokenGenerationIncludesRequiredClaims(): void {
    // Test custom logic, not JWT library
    $manager = $this->createManager();
    $token = $manager->initialize($capabilities);

    $decoded = $this->decodeWithoutValidation($token);
    $this->assertArrayHasKey('capabilities', $decoded);
    $this->assertEquals(['sampling' => true], $decoded['capabilities']);
  }

  public function testCapabilityValidation(): void {
    // Test business logic around capabilities
    $session = new SessionContext(['sampling' => true], 'jti-123', time() + 3600);
    $this->assertTrue($session->hasCapability('sampling'));
    $this->assertFalse($session->hasCapability('elicitation'));
  }

  // Do NOT test: JWT signature verification (firebase/php-jwt responsibility)
  // Do NOT test: Key loading (simple_oauth responsibility)
}
```

**Coordinator Tests**:
```php
class SamplingCoordinatorTest extends UnitTestCase {

  public function testRejectsRequestWithoutCapability(): void {
    $session = new SessionContext([], 'jti', time() + 3600);
    $coordinator = $this->createCoordinator();

    $this->expectException(\RuntimeException::class);
    $coordinator->requestCompletion($instruction, $session);
  }

  public function testCorrelationIdGeneration(): void {
    // Verify UUID v4 format
    $coordinator = $this->createCoordinator();
    $result = $coordinator->requestCompletion($instruction, $validSession);

    $this->assertMatchesRegularExpression(
      '/^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/',
      $result->params['correlationId']
    );
  }

  // Do NOT test: Repository database operations (kernel test responsibility)
}
```

**Value Object Tests**:
```php
class ElicitationPromptTest extends UnitTestCase {

  public function testRejectsInvalidPromptType(): void {
    $this->expectException(\InvalidArgumentException::class);
    new ElicitationPrompt('invalid-type', 'Question?');
  }

  public function testSelectPromptRequiresOptions(): void {
    $this->expectException(\InvalidArgumentException::class);
    ElicitationPrompt::select('Question?', []); // Empty options
  }

  public function testSerializationFormat(): void {
    $prompt = ElicitationPrompt::text('Name?', 'default', 'placeholder');
    $array = $prompt->toArray();

    $this->assertEquals('text', $array['type']);
    $this->assertEquals('Name?', $array['question']);
    $this->assertArrayNotHasKey('options', $array); // Null filtered out
  }
}
```

### Kernel Test Coverage

**Repository Tests** (`tests/src/Kernel/PendingRequest/PendingRequestRepositoryTest.php`):
```php
class PendingRequestRepositoryTest extends KernelTestBase {

  protected static $modules = ['mcp_server'];

  public function testCreateAndRetrievePendingRequest(): void {
    $envelope = new RequestEnvelope(
      'corr-123',
      'jti-456',
      'sampling',
      ['key' => 'value'],
      new \DateTimeImmutable('+60 seconds')
    );

    $requestId = $this->repository->createPending($envelope);
    $this->assertInstanceOf(PendingRequestId::class, $requestId);

    // Verify database row
    $row = $this->findRowByCorrelationId('corr-123');
    $this->assertEquals('pending', $row['status']);
    $this->assertEquals('jti-456', $row['session_jti']);
  }

  public function testStoreResponseUpdatesStatus(): void {
    $requestId = $this->createPendingRequest();

    $response = new ResponseEnvelope(['result' => 'data']);
    $this->repository->storeResponse($requestId, $response);

    $row = $this->findRowById($requestId->value());
    $this->assertEquals('completed', $row['status']);
    $this->assertNotNull($row['response_payload']);
  }

  public function testAwaitResponseReturnsOnCompletion(): void {
    $requestId = $this->createPendingRequest();

    // Simulate async response in separate "process"
    $this->storeResponseInBackground($requestId);

    $deadline = new \DateTimeImmutable('+5 seconds');
    $response = $this->repository->awaitResponse($requestId, $deadline);

    $this->assertNotNull($response);
    $this->assertInstanceOf(ResponseEnvelope::class, $response);
  }

  public function testAwaitResponseTimesOut(): void {
    $requestId = $this->createPendingRequest();

    // Short deadline, no response
    $deadline = new \DateTimeImmutable('+1 second');
    $response = $this->repository->awaitResponse($requestId, $deadline);

    $this->assertNull($response);

    // Verify marked expired
    $row = $this->findRowById($requestId->value());
    $this->assertEquals('expired', $row['status']);
  }

  public function testExpireOverdueMarksExpired(): void {
    // Create request that already expired
    $envelope = new RequestEnvelope(
      'corr-old',
      'jti-123',
      'sampling',
      [],
      new \DateTimeImmutable('-60 seconds') // Past
    );
    $this->repository->createPending($envelope);

    $count = $this->repository->expireOverdue();
    $this->assertEquals(1, $count);

    $row = $this->findRowByCorrelationId('corr-old');
    $this->assertEquals('expired', $row['status']);
  }

  public function testConcurrentStoreResponseUsesLocking(): void {
    // Test transaction isolation
    // This is complex - consider documenting expected behavior
    // rather than testing at this level
  }

  // Helper methods
  private function createPendingRequest(): PendingRequestId { /* ... */ }
  private function findRowByCorrelationId(string $id): array { /* ... */ }
  private function storeResponseInBackground(PendingRequestId $id): void { /* ... */ }
}
```

**Service Wiring Tests** (`tests/src/Kernel/ServiceWiringTest.php`):
```php
class ServiceWiringTest extends KernelTestBase {

  protected static $modules = ['mcp_server', 'simple_oauth'];

  public function testAllServicesWiredCorrectly(): void {
    $services = [
      'mcp_server.jwt_session_manager',
      'mcp_server.pending_request_repository',
      'mcp_server.sampling_coordinator',
      'mcp_server.elicitation_coordinator',
      'mcp_server.streaming_response_emitter',
    ];

    foreach ($services as $serviceId) {
      $service = $this->container->get($serviceId);
      $this->assertNotNull($service, "Service {$serviceId} should be available");
    }
  }

  public function testConfigurationSchemaValid(): void {
    $config = $this->config('mcp_server.settings');
    $this->assertIsInt($config->get('session.jwt_ttl'));
    $this->assertIsString($config->get('session.issuer'));
    $this->assertIsInt($config->get('pending.poll_interval_ms'));
  }
}
```

### Functional Test (ONE class, ONE method)

**End-to-End Flow Test** (`tests/src/Functional/McpSamplingElicitationTest.php`):
```php
/**
 * Tests MCP sampling and elicitation end-to-end flows.
 *
 * @group mcp_server
 */
class McpSamplingElicitationTest extends BrowserTestBase {

  protected static $modules = ['mcp_server', 'simple_oauth'];

  protected $defaultTheme = 'stark';

  /**
   * Tests complete sampling and elicitation workflows.
   */
  public function testSamplingAndElicitationFlows(): void {
    // Single comprehensive test method using helper methods
    $this->doTestSamplingFlow();
    $this->doTestElicitationFlow();
    $this->doTestTimeoutScenario();
    $this->doTestSecurityIsolation();
  }

  // Helper methods for different scenarios
  private function doTestSamplingFlow(): void {
    // 1. Initialize session with JWT
    $token = $this->initializeSession(['sampling' => true]);

    // 2. Call tool that triggers sampling
    $response = $this->callTool('analyze_data', ['data' => '...'], $token);

    // 3. Verify SSE stream started
    $this->assertStringContains('event: request', $response);
    $this->assertStringContains('sampling/createMessage', $response);

    // 4. Extract correlation ID
    $correlationId = $this->extractCorrelationId($response);

    // 5. Send client response
    $clientResponse = $this->sendSamplingResponse($correlationId, $token);
    $this->assertEquals(202, $clientResponse->getStatusCode());

    // 6. Verify tool completes with result
    // (In real test, would poll SSE stream for completion)
  }

  private function doTestElicitationFlow(): void {
    $token = $this->initializeSession(['elicitation' => true]);

    // Tool requests user input
    $response = $this->callTool('configure_export', [], $token);
    $this->assertStringContains('prompts/get', $response);

    $correlationId = $this->extractCorrelationId($response);

    // User selects option
    $clientResponse = $this->sendElicitationResponse(
      $correlationId,
      ['value' => 'json'],
      $token
    );
    $this->assertEquals(202, $clientResponse->getStatusCode());
  }

  private function doTestTimeoutScenario(): void {
    // Set very short timeout for testing
    $this->config('mcp_server.settings')
      ->set('pending.sampling_timeout', 2)
      ->save();

    $token = $this->initializeSession(['sampling' => true]);
    $response = $this->callTool('slow_analysis', [], $token);

    // Don't send response - let it timeout
    // Verify timeout error in SSE stream
    $this->assertStringContains('event: error', $response);
    $this->assertStringContains('timed out', $response);
  }

  private function doTestSecurityIsolation(): void {
    // Create two sessions
    $token1 = $this->initializeSession(['sampling' => true]);
    $token2 = $this->initializeSession(['sampling' => true]);

    // Start request with session 1
    $response = $this->callTool('analyze', [], $token1);
    $correlationId = $this->extractCorrelationId($response);

    // Attempt to respond with session 2
    $clientResponse = $this->sendSamplingResponse($correlationId, $token2);
    $this->assertEquals(403, $clientResponse->getStatusCode());

    // Verify error logged
    $logs = $this->getRecentLogs('mcp_server');
    $this->assertStringContains('injection attempt', $logs);
  }

  // Helper methods for test utilities
  private function initializeSession(array $capabilities): string { /* ... */ }
  private function callTool(string $name, array $params, string $token): string { /* ... */ }
  private function extractCorrelationId(string $sseResponse): string { /* ... */ }
  private function sendSamplingResponse(string $correlationId, string $token): Response { /* ... */ }
  private function sendElicitationResponse(string $correlationId, array $data, string $token): Response { /* ... */ }
}
```

### Security Test Coverage

**Session Isolation Tests**:
- Cross-session response injection (different JTI)
- Expired token handling
- Missing capability enforcement
- Correlation ID guessing attempts

**Timing Attack Tests**:
```php
public function testJtiComparisonIsConstantTime(): void {
  // Measure timing variance for valid vs invalid JTI
  // Should not have significant difference
  $validTimes = [];
  $invalidTimes = [];

  for ($i = 0; $i < 100; $i++) {
    $start = microtime(true);
    $this->handleResponse($validJti);
    $validTimes[] = microtime(true) - $start;

    $start = microtime(true);
    $this->handleResponse($invalidJti);
    $invalidTimes[] = microtime(true) - $start;
  }

  $validAvg = array_sum($validTimes) / count($validTimes);
  $invalidAvg = array_sum($invalidTimes) / count($invalidTimes);

  // Timing difference should be negligible (< 5%)
  $this->assertLessThan(0.05, abs($validAvg - $invalidAvg) / $validAvg);
}
```

### Test Data Generators

**Mock MCP Clients**:
```php
class MockMcpClient {
  public function initialize(array $capabilities): string {
    // Return mock JWT
  }

  public function callTool(string $name, array $params, string $token): array {
    // Simulate tool call
  }

  public function handleSamplingRequest(array $request): array {
    // Simulate LLM completion
    return [
      'role' => 'assistant',
      'content' => ['type' => 'text', 'text' => 'Mock analysis result'],
      'model' => 'mock-model',
    ];
  }

  public function handleElicitationRequest(array $request): array {
    // Simulate user input
    return ['value' => 'mock-selection'];
  }
}
```

### Testing Documentation

**TESTING.md**:
```markdown
# Testing MCP Sampling and Elicitation

## Running Tests

```bash
# All tests
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/

# Unit tests only (fast)
vendor/bin/phpunit --testsuite=unit web/modules/contrib/mcp_server/tests/

# Kernel tests
vendor/bin/phpunit --testsuite=kernel web/modules/contrib/mcp_server/tests/

# Functional test
vendor/bin/phpunit web/modules/contrib/mcp_server/tests/src/Functional/
```

## Manual Testing with MCP Inspector

1. Install MCP Inspector: [link]
2. Configure endpoint: `https://your-drupal.local/mcp/sse`
3. Initialize with capabilities:
   ```json
   {
     "capabilities": {
       "sampling": {},
       "elicitation": {}
     }
   }
   ```
4. Call tool that triggers sampling/elicitation
5. Observe SSE stream
6. Respond to request
7. Verify tool completion

## Test Coverage Goals

- Unit tests: 80%+ coverage of business logic
- Kernel tests: 90%+ coverage of database operations
- Functional test: Critical happy paths and security boundaries
- Focus on meaningful tests, not framework coverage
```

### Performance Testing
```php
public function testPollingPerformanceUnderLoad(): void {
  // Create 10 concurrent pending requests
  $requests = [];
  for ($i = 0; $i < 10; $i++) {
    $requests[] = $this->createPendingRequest();
  }

  // Measure poll timing
  $start = microtime(true);
  $this->repository->awaitResponse($requests[0], new \DateTimeImmutable('+5 seconds'));
  $duration = microtime(true) - $start;

  // Should complete in < 1 second with proper indexing
  $this->assertLessThan(1.0, $duration);
}
```

### Cross-Database Testing
If testing with multiple databases:
```php
class CrossDatabaseTest extends KernelTestBase {
  public function testBlobSerializationPortability(): void {
    // Test JSON serialization works across MySQL, MariaDB, PostgreSQL
    // Focus on blob storage and retrieval
  }
}
```

</details>
