---
id: 17
summary: "Replace JWT-based session management with database-backed session storage, eliminating JwtSessionManager entirely"
created: 2025-11-25
updated: 2025-11-25
---

# Plan: Database-Backed Session Storage for MCP Server

## Plan Clarifications

| Question | Answer | Date |
|----------|--------|------|
| **Session creation flow**: When and how are sessions created? | Per MCP spec, server generates session ID during `initialize` request and returns it in `Mcp-Session-Id` response header. Client includes `Mcp-Session-Id: <session_id>` on all subsequent requests. See [MCP Spec - Session Management](https://modelcontextprotocol.io/specification/draft/basic/transports#session-management). | 2025-11-25 |
| **MCP session headers**: What headers do clients send for session management? | Per MCP spec, clients include `Mcp-Session-Id: <session_id>` and `MCP-Protocol-Version: <version>` on all requests after initialization. Server validates these headers. | 2025-11-25 |
| **mcp_pending_request table migration**: How to handle `session_jti` column rename? | No migration needed - plan not yet implemented. Use `session_id` column name directly in new schema. | 2025-11-25 |
| **Session TTL source**: Where does `expires_at` come from? | From existing config `mcp_server.settings:session.jwt_ttl` (rename to `session.ttl`). Default 86400 seconds (24 hours). | 2025-11-25 |
| **UUID collision handling**: Should `session_id` have UNIQUE constraint? | Yes, add UNIQUE constraint. Assume UUID v4 collision impossible in practice (2^122 keyspace). | 2025-11-25 |

## Original Work Order

> It looks like creating a JWT-based session tracking for the MCP session was not a good idea. Instead, we should move to a database session tracking.

## Executive Summary

This plan eliminates the `JwtSessionManager` class entirely and replaces it with a database-backed session management system. JWT tokens are NOT needed for MCP sessions - OAuth2 (via simple_oauth) already handles Drupal authentication, and these concerns are orthogonal.

The solution implements two database components:
1. **DatabaseSessionStore**: Persistent storage for MCP Server Protocol's message queue functionality (implements `SessionStoreInterface` from mcp/sdk)
2. **DbSessionManager**: Generates and validates session metadata (session IDs, filesystem roots, expiry) stored in database

Session IDs are randomly generated UUIDs (not JWT jti claims). Session metadata (roots, expiry) is stored per session in the database. Capabilities (sampling, elicitation) are removed entirely - they are always available as core protocol features.

This architecture enables horizontal scaling, improves performance over the current file-based workaround (~2ms vs ~5ms per operation), maintains proper session isolation, and simplifies the codebase by removing unnecessary JWT complexity.

## Context

### Current State vs Target State

| Current State | Target State | Why? |
|--------------|--------------|------|
| `JwtSessionManager` creates JWT tokens and validates them | `DbSessionManager` generates random UUIDs and validates against database | JWT is unnecessary - OAuth2 already handles authentication |
| `JwtSessionManager` implements `SessionStoreInterface` with in-memory storage | `DatabaseSessionStore` implements `SessionStoreInterface` with database persistence | In-memory storage doesn't survive HTTP request boundaries, causing session lookup failures |
| Session data lost between requests, causing 404 "Session not found" errors | Session data persists across requests in database | Enable proper MCP protocol functionality for listing tools/prompts/resources |
| Using `FileSessionStore` as temporary workaround (~5ms per operation) | Using `DatabaseSessionStore` (~2ms per operation) | Improve performance and scalability |
| File-based storage in `sites/default/files/private/mcp_sessions/` | Database tables: `mcp_session_metadata` and `mcp_session_queue` | Better for containerized/distributed deployments |
| No visibility into session state | Can query session data directly from database | Improve observability and debugging |
| JWT tokens carry capabilities (sampling/elicitation) | Capabilities removed entirely - always enabled | Sampling and elicitation are core protocol features, not optional |
| Session context includes capability checks | Session context simplified to session_id, roots, expiry | Remove unnecessary complexity |
| JWT jti claim used for session isolation | Random UUID session_id used for session isolation | Simpler architecture without JWT dependency |

### Background

The MCP Server Protocol library (`mcp/sdk`) requires a `SessionStoreInterface` implementation to manage:
- **Outgoing message queues**: Messages waiting to be sent to clients via SSE
- **Pending requests**: Server-initiated requests (sampling/elicitation) awaiting client responses
- **Response storage**: Client responses indexed by request ID

Initially, `JwtSessionManager` was created under the mistaken assumption that JWT tokens were needed for MCP session management. This created two problems:

1. **Architectural confusion**: JWT tokens were conflated with OAuth2 authentication (which is orthogonal)
2. **In-memory storage bug**: The manager included `private array $sessionData = []` for the MCP Protocol's message queuing needs, but in-memory storage doesn't survive HTTP request boundaries

When the MCP Protocol tries to retrieve session data on subsequent requests, the in-memory array is empty, causing 404 errors with "Session not found or has expired."

A file-based session store (`FileSessionStore`) was implemented as a quick fix, but introduces performance overhead (~5ms per operation) and deployment complexity in containerized environments.

Additionally, `JwtSessionManager` carried "capabilities" (sampling, elicitation) in JWT tokens, adding unnecessary complexity for features that are always available as core protocol functionality.

**Key insight**: MCP sessions only need:
- A unique session identifier (random UUID)
- Filesystem roots for security
- Expiry timestamps
- Message queue persistence

No JWT tokens are required. Session management is orthogonal to authentication.

## Architectural Approach

```mermaid
graph TB
    Client[MCP Client]
    Controller[McpServerController]
    Protocol[MCP Protocol Library]
    DbSession[DbSessionManager]
    DBStore[DatabaseSessionStore]
    MetaTable[(mcp_session_metadata)]
    QueueTable[(mcp_session_queue)]

    Client -->|"HTTP Headers:<br/>Mcp-Session-Id: &lt;uuid&gt;<br/>MCP-Protocol-Version: &lt;version&gt;"| Controller
    Controller -->|"1. Validate Protocol Version"| Controller
    Controller -->|"2. Validate Session ID"| DbSession
    DbSession -->|Query Metadata| MetaTable
    DbSession -->|Return SessionContext| Controller
    Controller -->|Process Request| Protocol
    Protocol -->|Read/Write Queue| DBStore
    DBStore -->|SQL Operations| QueueTable

    style DbSession fill:#e1f5ff
    style DBStore fill:#d4edda
    style MetaTable fill:#d4edda
    style QueueTable fill:#d4edda
```

### Component 1: Database Schema Definition

**Objective**: Create two dedicated database tables to store session metadata and MCP Protocol message queue data with proper indexing for performance and automatic expiration.

#### Table 1: `mcp_session_metadata`

Stores session metadata for validation and security. Schema design:

- **Primary key**: `session_id` (VARCHAR 36) - Randomly generated UUID v4 per MCP spec requirements
- **UNIQUE constraint**: On `session_id` column (UUID v4 collision assumed impossible with 2^122 keyspace)
- **Filesystem roots**: `roots` (TEXT) - JSON-encoded array of allowed filesystem paths
- **Creation time**: `created_at` (BIGINT) - Unix timestamp when session was created
- **Expiration**: `expires_at` (BIGINT) - Unix timestamp for automatic cleanup (from `session.ttl` config)
- **Last activity**: `last_activity` (BIGINT) - Unix timestamp for monitoring
- **Index on `expires_at`**: Optimize garbage collection queries

Example row:
```
session_id: "550e8400-e29b-41d4-a716-446655440000"
roots: ["public://", "private://"]
created_at: 1732550000
expires_at: 1732636400
last_activity: 1732550123
```

#### Table 2: `mcp_session_queue`

Stores serialized MCP Protocol message queue data. Schema design:

- **Primary key**: `session_id` (VARCHAR 36) - References session metadata
- **Session data**: `data` (BLOB) - Serialized session state from MCP Protocol
- **Activity tracking**: `last_activity` (BIGINT) - Unix timestamp for monitoring
- **Expiration**: `expires_at` (BIGINT) - Unix timestamp for automatic cleanup
- **Index on `expires_at`**: Optimize garbage collection queries

This table is separate from `mcp_pending_request` which tracks individual async requests. The session queue stores the MCP Protocol's internal state machine data (outgoing messages, pending requests, client responses).

### Component 2: Session Management Implementation

**Objective**: Implement two complementary services - one for session metadata management (DbSessionManager) and one for MCP Protocol message queue storage (DatabaseSessionStore).

#### Part A: DbSessionManager

Replaces `JwtSessionManager` with database-backed session management. This service:

**Core Responsibilities:**
- Generates random UUID v4 session IDs during MCP `initialize` request
- Stores session metadata (session_id, roots, created_at, expires_at) in `mcp_session_metadata` table
- Validates session IDs from `Mcp-Session-Id` request headers (per MCP spec)
- Returns `SessionContext` objects for validated sessions
- Updates `last_activity` timestamps on each request

**Key Methods:**
- **`createSession(array $roots = []): string`**: Generates random UUID v4, calculates expiry from `session.ttl` config (default 86400s), stores metadata, returns session_id
- **`validateSession(string $sessionId): SessionContext`**: Queries database, validates expiry, returns context (throws `SessionValidationException` if expired/not found)
- **`updateActivity(string $sessionId): void`**: Updates last_activity timestamp
- **`destroySession(string $sessionId): bool`**: Deletes session metadata (called on HTTP DELETE)
- **`gc(): int`**: Removes expired sessions, returns count

**Session Creation Flow (per MCP spec):**
1. Client sends `initialize` request without `Mcp-Session-Id` header
2. Controller calls `createSession()` to generate UUID v4 and store metadata
3. Server returns `InitializeResult` with `Mcp-Session-Id: <session_id>` header
4. Client includes `Mcp-Session-Id: <session_id>` on all subsequent requests
5. Controller extracts session ID from header and calls `validateSession()`

**Configuration:**
- TTL from `mcp_server.settings:session.ttl` (renamed from `session.jwt_ttl`)
- Default: 86400 seconds (24 hours)

**SessionContext Changes:**
- Replace `jti` property with `sessionId` (string)
- Remove `capabilities` property (no longer needed)
- Remove `issuer` and `audience` properties (JWT-specific)
- Keep `roots` property (filesystem security)
- Keep `expiry` property (session validation)

**Security Considerations:**
- Session IDs must be cryptographically random (use `Uuid::uuid4()`)
- Constant-time comparison for session validation
- Automatic expiry validation on every request
- Isolation validation for pending requests (session_id matching)

#### Part B: DatabaseSessionStore

Implements `SessionStoreInterface` from mcp/sdk for MCP Protocol message queue persistence.

**Implementation Approach:**

Custom implementation directly with database persistence:
- **`exists(Uuid $id): bool`**: Query `mcp_session_queue` for non-expired session record
- **`read(Uuid $id): string|false`**: Fetch serialized session data, return FALSE if not found
- **`write(Uuid $id, string $data): bool`**: Use `MERGE` (upsert) to insert or update session data with updated timestamps
- **`destroy(Uuid $id): bool`**: Delete session record by UUID
- **`gc(): array`**: Delete expired sessions and return count of removed records

**Implementation Details:**
- Database connection failures handled gracefully
- Proper escaping of binary data in BLOB fields
- Atomic operations to prevent race conditions
- Logging for observability
- Performance target: <2ms per operation

### Component 3: Service Container Configuration

**Objective**: Update Drupal service definitions to use `DbSessionManager` and `DatabaseSessionStore`, removing all JWT-related services.

Changes required in `mcp_server.services.yml`:

1. **Remove `mcp_server.jwt_session_manager` service entirely**:
   - This service is no longer needed
   - JWT tokens are not used for MCP sessions
   - OAuth2 (via simple_oauth) handles authentication

2. **Add `mcp_server.db_session_manager` service**:
   - Class: `Drupal\mcp_server\Session\DbSessionManager`
   - Dependencies: `@database`, `@config.factory`, `@logger.channel.mcp_server`
   - Replaces JWT manager for session validation

3. **Replace `mcp_server.session.store` service**:
   - Change class from `FileSessionStore` to `Drupal\mcp_server\Session\DatabaseSessionStore`
   - Dependencies: `@database`, `@config.factory`, `@logger.channel.mcp_server`
   - Remove file directory parameter

4. **Update service dependencies**:
   - `mcp_server.server.factory`: Already references `@mcp_server.session.store` (no change)
   - `McpServerController`: Replace `@mcp_server.jwt_session_manager` with `@mcp_server.db_session_manager`
   - `mcp_server.sampling_coordinator`: Replace `JwtSessionManager` dependency with `DbSessionManager` in constructor
   - `mcp_server.elicitation_coordinator`: Replace `JwtSessionManager` dependency with `DbSessionManager` in constructor

### Component 4: Cron-Based Garbage Collection

**Objective**: Implement automatic cleanup of expired session data from both tables via Drupal's cron system to prevent database bloat.

Add a cron hook implementation:

- **Hook**: `hook_cron()`
- **Actions**:
  - Call `DbSessionManager::gc()` to clean up expired session metadata
  - Call `DatabaseSessionStore::gc()` to clean up expired message queue data
- **Frequency**: Every cron run (typically 3 hours default, configurable)
- **Logging**: Record count of expired sessions removed from each table
- **Performance**: Indexed queries on `expires_at` columns for efficiency

The garbage collection query patterns:
```sql
-- Clean session metadata
DELETE FROM mcp_session_metadata WHERE expires_at < UNIX_TIMESTAMP()

-- Clean message queue data
DELETE FROM mcp_session_queue WHERE expires_at < UNIX_TIMESTAMP()
```

**Note**: Orphaned queue records (where session metadata was deleted but queue wasn't) will be cleaned up by their own expiry. Consider adding cascade delete or periodic orphan cleanup if needed.

### Component 5: Test Coverage

**Objective**: Ensure both session management components work correctly through kernel tests that validate core functionality without testing framework internals.

#### Test Class 1: `DbSessionManagerTest`

Test scenarios for session metadata management:

1. **Session creation**: `createSession()` generates random UUID and stores metadata
2. **Session validation**: `validateSession()` returns valid SessionContext for active sessions
3. **Session expiration**: Expired sessions throw `SessionValidationException`
4. **Session isolation**: Multiple sessions with different IDs don't interfere
5. **Activity updates**: `updateActivity()` updates last_activity timestamp
6. **Session destruction**: `destroySession()` removes metadata
7. **Garbage collection**: Expired sessions removed, active ones remain
8. **Filesystem roots**: SessionContext correctly carries roots from database
9. **Remove capability tests**: No tests for capabilities (removed from SessionContext)

#### Test Class 2: `DatabaseSessionStoreTest`

Test scenarios for MCP Protocol message queue:

1. **Session persistence**: Write session data, read it back in same request
2. **Session expiration**: Verify expired sessions return FALSE on read
3. **Session isolation**: Multiple sessions with different UUIDs don't interfere
4. **Garbage collection**: Expired sessions are removed, active ones remain
5. **Error handling**: Database failures are logged and handled gracefully

Both test classes extend `KernelTestBase`:
- Install schema during setup
- Test ONLY the module's business logic (CRUD operations)
- Do NOT test database layer itself (framework responsibility)
- Use helper methods to consolidate expensive test setup

## Risk Considerations and Mitigation Strategies

<details>
<summary>Technical Risks</summary>

- **Database performance degradation under high session volume**: Frequent reads/writes could impact database performance
  - **Mitigation**: Add proper indexes on `expires_at`, use connection pooling, monitor query performance, consider adding Redis layer later if needed

- **Session data serialization compatibility**: PHP serialization format could break if internal MCP Protocol state structure changes
  - **Mitigation**: Document serialization format, add version checks, implement graceful fallback for unreadable session data

- **Transaction isolation issues**: Concurrent requests could cause race conditions
  - **Mitigation**: Use database transactions for write operations, implement optimistic locking if needed
</details>

<details>
<summary>Implementation Risks</summary>

- **Breaking existing sessions during migration**: Active sessions in file storage will be lost
  - **Mitigation**: Acceptable since module is in active development and not yet released; document that sessions will reset

- **Incomplete cleanup of file-based storage**: Old session files may remain
  - **Mitigation**: Document manual cleanup steps, consider adding update hook to remove old session directory

- **Service configuration errors**: Incorrect dependency injection could cause runtime failures
  - **Mitigation**: Clear cache after changes, test thoroughly, validate service definitions
</details>

<details>
<summary>Integration Risks</summary>

- **MCP Protocol library compatibility**: Changes could break assumptions in upstream library
  - **Mitigation**: Implement exact `SessionStoreInterface` contract, run existing functional tests

- **Breaking changes to coordinators**: SamplingCoordinator and ElicitationCoordinator depend on SessionContext structure
  - **Mitigation**: Update coordinators to use new SessionContext properties (sessionId instead of jti), remove capability checks

- **Breaking changes to controller**: McpServerController uses `getSession()` method that returns SessionContext
  - **Mitigation**: Update `getSession()` to use DbSessionManager, validate session IDs instead of JWT tokens
</details>

## Success Criteria

### Primary Success Criteria

1. **MCP clients can successfully list tools, prompts, and resources** without receiving "Session not found or has expired" errors
2. **Session data persists across HTTP requests** with read/write operations averaging <2ms
3. **All existing tests pass** with no regressions in functionality
4. **Cron garbage collection removes expired sessions** automatically without manual intervention
5. **Database queries are properly indexed** with no full table scans on session operations

## Resource Requirements

### Development Skills

- PHP 8.3+ with strong understanding of Drupal 11 dependency injection and service containers
- Database schema design and SQL optimization (indexes, BLOB handling)
- Drupal hook system (cron, schema, install/update hooks)
- PHPUnit kernel testing patterns
- Understanding of MCP Server Protocol session lifecycle

### Technical Infrastructure

- Existing Drupal 11.1 installation with database access
- Development environment with cache clearing capabilities
- PHPUnit test runner configured
- Database administration tools for schema inspection

## Integration Strategy

This work integrates with existing MCP Server components:

1. **Controller layer changes required**: `McpServerController` needs to:
   - Replace `JwtSessionManager` dependency with `DbSessionManager`
   - Add session creation during `initialize` handling (call `createSession()`, return session ID in `Mcp-Session-Id` response header)
   - Update session validation to:
     - Extract session ID from `Mcp-Session-Id` request header
     - Validate `MCP-Protocol-Version` header presence (per MCP spec)
     - Validate session ID via `DbSessionManager::validateSession()`
     - Return HTTP 400 if `Mcp-Session-Id` or `MCP-Protocol-Version` missing (except on initialize)
     - Return HTTP 404 if session expired/not found
   - Continue using session isolation validation for pending requests (using session_id instead of jti)
   - Handle HTTP DELETE requests to destroy sessions (implementation choice: always permit OR return HTTP 405 Method Not Allowed if policy disallows client termination)

2. **Coordinator updates required**: Both `SamplingCoordinator` and `ElicitationCoordinator` need to:
   - Replace `JwtSessionManager` dependency with `DbSessionManager`
   - Remove capability checks (`hasCapability()` calls) - sampling/elicitation always available
   - Update SessionContext usage (use `sessionId` property instead of `jti`)

3. **SessionContext breaking changes**:
   - Remove `jti`, `capabilities`, `issuer`, `audience` properties
   - Add `sessionId` property (string UUID)
   - Keep `roots` and `expiry` properties unchanged

4. **No API changes to MCP Protocol layer**: Implements existing `SessionStoreInterface` contract without modifications

5. **MCP protocol headers** _(per clarifications)_:
   - Every MCP request (except initialize) includes **two MCP headers**:
     - `Mcp-Session-Id: <session_id>` - For MCP session state management
     - `MCP-Protocol-Version: <version>` - For protocol version validation (per MCP spec)
   - Controller validates protocol version and session ID presence/validity
   - Returns HTTP 400 if headers missing, HTTP 404 if session expired

## Notes

**Module Development Status**: This module is in active development and has not been released. No backwards compatibility or migration support is required for existing sessions.

**JWT Removal**: JwtSessionManager, JWT token generation, and JWT validation are completely removed. MCP sessions use simple UUID-based session IDs.

**Capability Removal**: Sampling and elicitation capabilities are removed from the codebase. These are core MCP protocol features and are always available.

**Scope**: This plan addresses MCP session management ONLY. Authentication/authorization is handled separately and is not part of this plan.

**MCP Specification Compliance**: Session management follows the [MCP Streamable HTTP Transport specification](https://modelcontextprotocol.io/specification/draft/basic/transports#session-management):
- Sessions created during `initialize` request
- Session ID returned in `Mcp-Session-Id` response header
- Client includes `Mcp-Session-Id` header on all subsequent requests
- Client includes `MCP-Protocol-Version` header on all subsequent requests
- Server enforces both headers presence (HTTP 400 if missing, except on initialize)
- Server returns HTTP 404 for expired/invalid sessions
- Client can DELETE session explicitly via HTTP DELETE
- Server may return HTTP 405 Method Not Allowed if DELETE not permitted (implementation choice)

**MCP Protocol Headers**: MCP requests include two protocol-specific headers:
1. `Mcp-Session-Id: <session_id>` - MCP protocol state management
2. `MCP-Protocol-Version: <version>` - Protocol version compatibility

Controller validates both headers on each request (except initialize).

**Configuration Changes**:
- Rename `mcp_server.settings:session.jwt_ttl` to `session.ttl`
- Keep default 86400 seconds (24 hours)
- Remove JWT-specific configs (issuer, algorithm, key paths)

**Database Schema Notes**:
- `mcp_session_metadata`: UUID v4 with UNIQUE constraint (collision assumed impossible)
- `mcp_pending_request.session_jti`: Rename to `session_id` (no migration - not yet implemented)
- Foreign keys not required - orphaned records cleaned by expiry

**MCP SDK Resources**: The `mcp/sdk` package provides a `Psr16StoreSession` class that implements `SessionStoreInterface` using PSR-16 SimpleCache. The SDK also includes reference implementations: `FileSessionStore` and `InMemorySessionStore` in the `Mcp\Server\Session` namespace. We're implementing DatabaseSessionStore directly for better control.

**File Cleanup**: After implementation, the `sites/default/files/private/mcp_sessions/` directory can be removed manually. Consider adding an update hook for automatic cleanup.

**Future Optimization**: If database performance becomes a bottleneck at scale, a Redis-backed session store can be implemented following the same `SessionStoreInterface` contract without changes to the MCP Server factory or controller layers.

## Task Dependencies

```mermaid
graph TD
    001[Task 001: Database Schema & Services] --> 002[Task 002: Refactor SessionContext]
    001 --> 003[Task 003: Update Controller & Coordinators]
    002 --> 003
    003 --> 004[Task 004: Service Configuration]
    004 --> 005[Task 005: Integration Tests]
```

## Execution Blueprint

**Validation Gates:**
- Reference: `.ai/task-manager/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Foundation Layer
**Parallel Tasks:**
- ✔️ Task 001: Database Schema and Session Services (database tables, DbSessionManager, DatabaseSessionStore)

**Complexity:** 5.0 (Drupal schema API + database design + 2 service implementations)

### ✅ Phase 2: Data Model Refactoring
**Parallel Tasks:**
- ✔️ Task 002: Refactor SessionContext to Remove JWT Dependencies (depends on: 001)

**Complexity:** 3.0 (Straightforward property removal and constructor update)

### ✅ Phase 3: Request Handling Layer
**Parallel Tasks:**
- ✔️ Task 003: Update Controller and Coordinators for Database Sessions (depends on: 001, 002)

**Complexity:** 5.0 (MCP header validation, controller logic, 2 coordinator updates)

### ✅ Phase 4: Service Configuration
**Parallel Tasks:**
- ✔️ Task 004: Service Configuration and JWT Service Removal (depends on: 001, 002, 003)

**Complexity:** 3.2 (YAML configuration, service replacement)

### ✅ Phase 5: Validation and Quality Assurance
**Parallel Tasks:**
- ✔️ Task 005: Integration Tests for Database Session Management (depends on: 001, 002, 003, 004)

**Complexity:** 5.0 (Kernel tests for multiple components, database assertions)

### Post-phase Actions

After Phase 1 completion:
- Verify database tables created correctly: `vendor/bin/drush sql:query "SHOW TABLES LIKE 'mcp_session%'"`
- Verify services registered: `vendor/bin/drush debug:container mcp_server.db_session_manager`

After Phase 5 completion:
- Run full test suite: `vendor/bin/phpunit --testsuite=kernel`
- Verify garbage collection: `vendor/bin/drush cron`
- Manual cleanup of old file-based sessions: `rm -rf sites/default/files/private/mcp_sessions/`

### Execution Summary
- Total Phases: 5
- Total Tasks: 5
- Maximum Parallelism: 1 task per phase (linear dependency chain)
- Critical Path Length: 5 phases
- Average Task Complexity: 4.24/10

### Change Log

- **2025-11-25**: Plan refinement session
  - Added "Plan Clarifications" table documenting session creation flow, MCP headers, TTL config, and UUID constraints
  - Updated Component 1 schema with UNIQUE constraint details and config source for expiry
  - Updated Component 2 (DbSessionManager) with complete session creation flow per MCP spec
  - Updated Integration Strategy with MCP protocol header requirements
  - Added MCP spec compliance notes and configuration change details
  - Clarified `mcp_pending_request.session_jti` rename (no migration needed)

- **2025-11-25**: MCP spec validation
  - Validated plan against [MCP Streamable HTTP Transport specification](https://modelcontextprotocol.io/specification/draft/basic/transports#session-management)
  - Added `MCP-Protocol-Version` header validation requirement to Integration Strategy
  - Added HTTP 405 Method Not Allowed option for DELETE requests
  - Updated MCP Specification Compliance section with complete requirements list

- **2025-11-25**: Scope refinement
  - Removed all authentication/OAuth2 references - this plan addresses session management ONLY
  - Removed Task Dependencies and Execution Blueprint sections (will be regenerated during task generation)
  - Updated clarifications table to focus on MCP protocol headers, not authentication
  - Simplified architectural diagram to show only MCP headers
  - Updated Integration Strategy to remove OAuth2 validation steps

- **2025-11-25**: Task generation complete
  - Added Task Dependencies diagram showing linear dependency chain
  - Added Execution Blueprint with 5 phases
  - Complexity analysis: All tasks scored ≤5.0 (no decomposition needed)
  - Average complexity: 4.24/10 across 5 tasks

---

**Note**: Manually archived on 2025-11-27
