---
id: 1
group: database-session-management
dependencies: []
status: completed
created: '2025-11-25'
skills:
  - drupal-backend
  - database
---
# Implement Database Schema and Session Management Services

## Objective
Create database schema for session storage and implement both `DatabaseSessionStore` and `DbSessionManager` services for persistent MCP session management.

## Skills Required
- **drupal-backend**: Drupal schema definition, service implementation, dependency injection
- **database**: SQL schema design, indexes, BLOB handling, upsert operations

## Acceptance Criteria
- [ ] `mcp_session_metadata` table created with proper indexes
- [ ] `mcp_session_queue` table created with proper indexes
- [ ] `DatabaseSessionStore` implements `SessionStoreInterface` from mcp/sdk
- [ ] `DbSessionManager` implements session CRUD operations
- [ ] Both services handle errors gracefully with logging
- [ ] Performance target <2ms per operation achieved

## Technical Requirements

### Database Schema

Create schema in `mcp_server.install`:

**Table 1: `mcp_session_metadata`**
- Primary key: `session_id` (VARCHAR 36)
- UNIQUE constraint on `session_id`
- `roots` (TEXT) - JSON-encoded filesystem roots array
- `created_at` (BIGINT) - Unix timestamp
- `expires_at` (BIGINT) - Unix timestamp
- `last_activity` (BIGINT) - Unix timestamp
- Index on `expires_at` for garbage collection

**Table 2: `mcp_session_queue`**
- Primary key: `session_id` (VARCHAR 36)
- `data` (BLOB) - Serialized MCP Protocol state
- `last_activity` (BIGINT) - Unix timestamp
- `expires_at` (BIGINT) - Unix timestamp
- Index on `expires_at` for garbage collection

### DatabaseSessionStore Service

Location: `src/Session/DatabaseSessionStore.php`

Implement `SessionStoreInterface` from mcp/sdk:
- `exists(Uuid $id): bool` - Check if session exists and not expired
- `read(Uuid $id): string|false` - Fetch serialized data from `mcp_session_queue`
- `write(Uuid $id, string $data): bool` - Upsert session data with updated timestamps
- `destroy(Uuid $id): bool` - Delete from `mcp_session_queue`
- `gc(): array` - Delete expired sessions, return removed count

Dependencies: `@database`, `@config.factory`, `@logger.channel.mcp_server`

### DbSessionManager Service

Location: `src/Session/DbSessionManager.php`

Methods:
- `createSession(array $roots = []): string` - Generate UUID v4, store in `mcp_session_metadata`, return session_id
- `validateSession(string $sessionId): SessionContext` - Query database, validate expiry, return context
- `updateActivity(string $sessionId): void` - Update `last_activity` timestamp
- `destroySession(string $sessionId): bool` - Delete from `mcp_session_metadata`
- `gc(): int` - Delete expired sessions, return count

Dependencies: `@database`, `@config.factory`, `@logger.channel.mcp_server`

Configuration: Read TTL from `mcp_server.settings:session.ttl` (default 86400)

## Input Dependencies
None - this is the foundation task

## Output Artifacts
- `mcp_server.install` with schema definitions
- `src/Session/DatabaseSessionStore.php`
- `src/Session/DbSessionManager.php`
- Both services registered but not yet wired into existing code

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Schema Implementation

Use Drupal's Schema API in `hook_schema()`:
```php
function mcp_server_schema() {
  $schema['mcp_session_metadata'] = [
    'fields' => [
      'session_id' => ['type' => 'varchar', 'length' => 36, 'not null' => TRUE],
      'roots' => ['type' => 'text', 'not null' => TRUE],
      'created_at' => ['type' => 'int', 'size' => 'big', 'not null' => TRUE],
      'expires_at' => ['type' => 'int', 'size' => 'big', 'not null' => TRUE],
      'last_activity' => ['type' => 'int', 'size' => 'big', 'not null' => TRUE],
    ],
    'primary key' => ['session_id'],
    'unique keys' => ['session_id' => ['session_id']],
    'indexes' => ['expires_at' => ['expires_at']],
  ];

  $schema['mcp_session_queue'] = [
    'fields' => [
      'session_id' => ['type' => 'varchar', 'length' => 36, 'not null' => TRUE],
      'data' => ['type' => 'blob', 'size' => 'big', 'not null' => TRUE],
      'last_activity' => ['type' => 'int', 'size' => 'big', 'not null' => TRUE],
      'expires_at' => ['type' => 'int', 'size' => 'big', 'not null' => TRUE],
    ],
    'primary key' => ['session_id'],
    'indexes' => ['expires_at' => ['expires_at']],
  ];

  return $schema;
}
```

### DatabaseSessionStore Implementation

```php
final class DatabaseSessionStore implements SessionStoreInterface {
  public function __construct(
    private readonly Connection $database,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {}

  public function exists(Uuid $id): bool {
    try {
      $result = $this->database->select('mcp_session_queue', 's')
        ->fields('s', ['session_id'])
        ->condition('session_id', $id->toString())
        ->condition('expires_at', time(), '>')
        ->execute()
        ->fetchField();
      return $result !== FALSE;
    } catch (\Exception $e) {
      $this->logger->error('Failed to check session existence: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  public function write(Uuid $id, string $data): bool {
    try {
      $ttl = $this->configFactory->get('mcp_server.settings')->get('session.ttl') ?? 86400;
      $this->database->merge('mcp_session_queue')
        ->key(['session_id' => $id->toString()])
        ->fields([
          'data' => $data,
          'last_activity' => time(),
          'expires_at' => time() + $ttl,
        ])
        ->execute();
      return TRUE;
    } catch (\Exception $e) {
      $this->logger->error('Failed to write session: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  // Implement remaining methods similarly
}
```

### DbSessionManager Implementation

Use `Uuid::uuid4()` for cryptographically secure UUID generation.
Store `roots` as JSON-encoded string.
Calculate expiry from config TTL.
Throw `SessionValidationException` for expired/not found sessions.

### Performance Considerations

- Use prepared statements (Drupal's query builder handles this)
- Index on `expires_at` enables fast garbage collection
- MERGE operation for atomic upserts
- Early return on expiry check

### Security Considerations

- Use constant-time comparison for session validation (hash_equals if needed)
- Validate session_id format (UUID v4 pattern)
- Log all failures for security monitoring
- Never expose session_id in error messages to clients

</details>
