---
id: 3
group: database-session-management
dependencies:
  - 1
  - 2
status: completed
created: '2025-11-25'
skills:
  - drupal-backend
  - php
---
# Update Controller and Coordinators for Database Sessions

## Objective
Update `McpServerController`, `SamplingCoordinator`, and `ElicitationCoordinator` to use database-backed sessions and validate MCP protocol headers.

## Skills Required
- **drupal-backend**: Drupal controller patterns, dependency injection, service wiring
- **php**: PHP 8.3+ refactoring, property access updates

## Acceptance Criteria
- [ ] `McpServerController` validates `Mcp-Session-Id` and `MCP-Protocol-Version` headers
- [ ] `McpServerController` returns HTTP 400 for missing headers (except on initialize)
- [ ] `McpServerController` uses `DbSessionManager` instead of `JwtSessionManager`
- [ ] `McpServerController` creates session during `initialize` request
- [ ] `McpServerController` returns `Mcp-Session-Id` header in initialize response
- [ ] `SamplingCoordinator` uses `sessionId` instead of `jti`
- [ ] `ElicitationCoordinator` uses `sessionId` instead of `jti`
- [ ] All capability checks removed from coordinators
- [ ] Controller returns HTTP 404 for expired/invalid sessions

## Technical Requirements

### McpServerController Updates

Location: `src/Controller/McpServerController.php`

#### Add MCP Header Validation

Add private method to validate protocol headers:

```php
private function validateMcpHeaders(Request $request, bool $requireSessionId = true): void {
  // Validate protocol version
  $protocolVersion = $request->headers->get('MCP-Protocol-Version');
  if (!$protocolVersion) {
    throw new BadRequestHttpException('Missing required header: MCP-Protocol-Version');
  }

  // Validate session ID (except on initialize)
  if ($requireSessionId) {
    $sessionId = $request->headers->get('Mcp-Session-Id');
    if (!$sessionId) {
      throw new BadRequestHttpException('Missing required header: Mcp-Session-Id');
    }
  }
}
```

#### Update initialize() Method

Add session creation and response header:

```php
public function initialize(Request $request): JsonResponse {
  $this->validateMcpHeaders($request, requireSessionId: false);

  // ... existing initialization logic ...

  // Create session
  $sessionId = $this->sessionManager->createSession($roots);

  $response = new JsonResponse($result);
  $response->headers->set('Mcp-Session-Id', $sessionId);

  return $response;
}
```

#### Update Other Route Methods

Add header validation to all route methods (handle, stream, etc.):

```php
public function handle(Request $request): JsonResponse {
  $this->validateMcpHeaders($request);

  $sessionId = $request->headers->get('Mcp-Session-Id');

  try {
    $context = $this->sessionManager->validateSession($sessionId);
  } catch (SessionValidationException $e) {
    throw new NotFoundHttpException('Session expired or invalid');
  }

  // ... rest of method using $context->sessionId instead of $context->jti ...
}
```

#### Update Constructor

Replace `JwtSessionManager` with `DbSessionManager`:

```php
public function __construct(
  // ... other dependencies ...
  private readonly DbSessionManager $sessionManager,
  // ... other dependencies ...
) {}
```

### SamplingCoordinator Updates

Location: `src/Session/SamplingCoordinator.php`

Update all references from `$context->jti` to `$context->sessionId`:

```php
public function createRequest(
  SessionContext $context,
  // ... other parameters ...
): string {
  $requestId = Uuid::v4()->toString();

  $this->database->insert('mcp_pending_request')
    ->fields([
      'request_id' => $requestId,
      'session_id' => $context->sessionId,  // Changed from jti
      // ... other fields ...
    ])
    ->execute();

  return $requestId;
}
```

Remove capability check if present:

```php
// REMOVE this:
if (!$context->hasCapability('sampling')) {
  throw new \Exception('Sampling not enabled');
}
```

### ElicitationCoordinator Updates

Location: `src/Session/ElicitationCoordinator.php`

Same changes as SamplingCoordinator:
- Replace `$context->jti` with `$context->sessionId`
- Remove capability checks

## Input Dependencies
- Task 1: `DbSessionManager` and `DatabaseSessionStore` services must exist
- Task 2: `SessionContext` must have `sessionId` property and removed JWT properties

## Output Artifacts
- Updated `src/Controller/McpServerController.php`
- Updated `src/Session/SamplingCoordinator.php`
- Updated `src/Session/ElicitationCoordinator.php`
- MCP header validation implemented
- HTTP 400 responses for missing headers
- HTTP 404 responses for invalid sessions

## Implementation Notes

<details>
<summary>Detailed Implementation Guide</summary>

### Header Validation Strategy

Per MCP spec, validate headers on EVERY request except `initialize`:

1. **initialize request**: Only validate `MCP-Protocol-Version`, do NOT require `Mcp-Session-Id`
2. **All other requests**: Validate both headers

Use `BadRequestHttpException` (HTTP 400) for missing headers.
Use `NotFoundHttpException` (HTTP 404) for expired/invalid sessions.

### Session Creation Flow

```php
// In initialize() method:
1. Validate MCP-Protocol-Version header
2. Parse request body and handle initialization
3. Call $this->sessionManager->createSession($roots)
4. Return response with Mcp-Session-Id header
```

### Session Validation Flow

```php
// In all other methods:
1. Validate both MCP headers
2. Extract session_id from header
3. Call $this->sessionManager->validateSession($sessionId)
4. Catch SessionValidationException -> throw NotFoundHttpException
5. Use $context->sessionId in coordinator calls
```

### Property Access Updates

Search for all occurrences of:
- `$context->jti` -> Replace with `$context->sessionId`
- `$context->hasCapability()` -> Remove these checks
- `$context->capabilities` -> Remove usage

### Error Handling

Be consistent with HTTP status codes:
- 400: Missing or malformed headers
- 404: Session expired or not found
- 405: DELETE not allowed (if implementing DELETE restrictions)

### Testing Considerations

Task 5 will add integration tests. For now, focus on:
- Correct dependency injection
- Proper exception handling
- Header validation on right methods

</details>
