---
id: 18
summary: "Migrate custom sampling/elicitation to SDK's ClientGateway and remove obsolete infrastructure"
created: 2025-11-26
---

# Plan: Migrate to SDK's ClientGateway for Sampling

## Original Work Order

> Migrate our custom sampling & elicitation implementation to use the SDK's client gateway. See PLAN_CLIENTGATEWAY_MIGRATION.md

## Executive Summary

This plan migrates the MCP server module from a custom `SamplingCoordinator`/`ElicitationCoordinator` implementation to the MCP SDK's built-in `ClientGateway` for sampling requests. The elicitation feature will be removed entirely as it's not supported by the SDK.

Both implementations are architecturally similar - they use blocking polling with `usleep()` and separate HTTP requests for responses. The SDK simply wraps this pattern in PHP 8.1+ Fibers for cleaner syntax. The existing `DatabaseSessionStore` already implements the SDK's `SessionStoreInterface`, enabling seamless database-backed state management.

This migration reduces code complexity from ~600 lines of custom code to ~50 lines of SDK integration, aligns with the official SDK patterns, and reduces long-term maintenance burden.

## Context

### Current State vs Target State

| Current State | Target State | Why? |
|--------------|--------------|------|
| Custom `SamplingCoordinator` with ~200 lines | SDK's `ClientGateway::sample()` | Reduces maintenance, uses official SDK |
| Custom `ElicitationCoordinator` | Removed entirely | Not supported by SDK, unused feature |
| `mcp_pending_request` database table | SDK uses `SessionStoreInterface` | Already implemented via `DatabaseSessionStore` |
| Custom `StreamingResult` detection in controller | SDK transport handles SSE | Simpler controller code |
| 13 custom PHP files for sampling/elicitation | 1 trait + handler modifications | ~92% reduction in custom code |
| Tools manually call `SamplingCoordinator` | Tools receive `ClientGateway` via injection | Cleaner API, SDK-standard pattern |

### Background

**Key Technical Finding**: The SDK does NOT require ReactPHP, Amp, or any event loop. It uses:
- **PHP 8.1+ native Fibers** for cooperative multitasking
- **Blocking polling** with `usleep(100000)` (100ms intervals)
- **Session storage** for cross-request communication

This matches our current custom implementation pattern exactly.

**Database Session Store**: The existing `DatabaseSessionStore` class already implements `SessionStoreInterface` and is wired into `McpServerFactory`. The SDK's `ClientGateway` will automatically use this for state management, ensuring persistence across PHP restarts and supporting multi-server deployments.

## Architectural Approach

```mermaid
flowchart TB
    subgraph Current["Current Architecture"]
        T1[Tool] --> SC[SamplingCoordinator]
        SC --> PR[(mcp_pending_request)]
        SC --> SR[StreamingResult]
        SR --> CTRL[Controller]
        CTRL --> SSE1[SSE Response]
    end

    subgraph Target["Target Architecture"]
        T2[Tool] --> CG[ClientGateway]
        CG --> FB[Fiber::suspend]
        FB --> TRANS[SDK Transport]
        TRANS --> DSS[(DatabaseSessionStore)]
        TRANS --> SSE2[SSE Response]
    end

    Current --> |"Migration"| Target
```

### SDK Integration

**Objective**: Wire the SDK's `ClientGateway` into the existing tool execution flow.

The `CustomCallToolHandler` currently calls `McpBridgeService::executeMcpTool()` without any sampling context. The handler must be modified to create a `ClientGateway` instance from the session and pass it to tool execution.

The SDK provides `ClientGateway` through session access. When a tool needs sampling, it calls `$gateway->sample()`, which suspends the current Fiber. The SDK transport detects this suspension and handles the SSE streaming automatically.

### Tool Sampling Trait

**Objective**: Provide a clean interface for tools that need sampling capabilities.

A `McpToolSamplingTrait` will provide `setClientGateway()` and a protected `sample()` method. Tools using this trait receive the gateway via setter injection during execution. This keeps the existing Tool API plugin architecture intact while adding sampling support.

### Infrastructure Removal

**Objective**: Delete all custom sampling and elicitation code that becomes obsolete.

Files to remove:
- **Sampling**: `SamplingCoordinator.php`, `SamplingInstruction.php`, `SamplingResponse.php`, `StreamingResult.php`
- **Elicitation**: `ElicitationCoordinator.php`, `ElicitationPrompt.php`, `ElicitationResponse.php`
- **Streaming**: `StreamingResult.php`, `StreamingResponseEmitter.php`
- **PendingRequest**: `PendingRequestId.php`, `PendingRequestRepository.php`, `RequestEnvelope.php`, `ResponseEnvelope.php`
- **Database**: Remove `mcp_pending_request` table schema from `mcp_server.install`
- **Services**: Remove related service definitions from `mcp_server.services.yml`

### Test Updates

**Objective**: Replace custom coordinator tests with SDK integration tests.

The existing `CoordinatorIntegrationTest` tests custom sampling/elicitation flows. These tests should be replaced with integration tests that verify `ClientGateway` injection works correctly and sampling requests flow through the SDK transport.

```mermaid
sequenceDiagram
    participant Tool
    participant Handler as CustomCallToolHandler
    participant Bridge as McpBridgeService
    participant Gateway as ClientGateway
    participant Session as DatabaseSessionStore
    participant Client as MCP Client

    Tool->>Handler: executeTool()
    Handler->>Gateway: create from session
    Handler->>Bridge: executeMcpTool(args, gateway)
    Bridge->>Tool: execute(args)
    Tool->>Gateway: sample(message)
    Gateway->>Gateway: Fiber::suspend()
    Gateway-->>Handler: suspended
    Handler-->>Client: SSE: sampling/createMessage
    Client->>Handler: POST: sampling response
    Handler->>Session: store response
    Session-->>Gateway: response found
    Gateway->>Gateway: Fiber::resume()
    Gateway-->>Tool: CreateSamplingMessageResult
    Tool-->>Handler: result
    Handler-->>Client: tool result
```

## Risk Considerations and Mitigation Strategies

<details>
<summary>Technical Risks</summary>

- **Client Compatibility**: MCP clients may not handle sampling responses correctly on SSE
    - **Mitigation**: Test with MCP Inspector first; verify Claude Desktop compatibility; document client requirements

- **Session Persistence**: SDK session behavior across PHP process restarts is unclear
    - **Mitigation**: Already using `DatabaseSessionStore` which persists to database; verify behavior in integration tests
</details>

<details>
<summary>Implementation Risks</summary>

- **Test Coverage Gaps**: Removing existing tests may leave gaps
    - **Mitigation**: Create SDK-based integration tests before removing old tests
</details>

## Success Criteria

### Primary Success Criteria

1. Tools can request sampling via `ClientGateway` injection and receive responses
2. All custom sampling/elicitation code removed (~600 lines eliminated)
3. `DatabaseSessionStore` correctly manages SDK session state across requests
4. Existing test suite passes (minus removed coordinator tests)
5. Sampling works end-to-end with MCP Inspector client

## Resource Requirements

### Development Skills

- PHP 8.1+ Fiber understanding
- MCP SDK familiarity (`ClientGateway`, `SessionStoreInterface`)
- Drupal service container and dependency injection

### Technical Infrastructure

- MCP PHP SDK (already installed via composer)
- MCP Inspector for integration testing
- PHPUnit for automated tests

## Integration Strategy

Tools requiring sampling will use the new trait pattern. The `CustomCallToolHandler` changes are internal and don't affect the external MCP protocol interface. No migration path is needed as this module is not yet released.

## Notes

- **Unreleased Module**: This module is not yet released, so no deprecation period or migration guides are needed - simply delete obsolete code
- The `DatabaseSessionStore` at `src/Session/DatabaseSessionStore.php` already implements `SessionStoreInterface` and is wired into `McpServerFactory` - no new session infrastructure needed
- The SDK's 100ms polling interval in `StreamableHttpTransport` matches our custom implementation's behavior
- Elicitation is intentionally removed entirely, not migrated, as the SDK doesn't support it

## Task Dependency Visualization

```mermaid
graph TD
    01[Task 01: Remove Obsolete Infrastructure] --> 02[Task 02: Create Sampling Trait]
    02 --> 03[Task 03: Update Handler/Bridge for Gateway]
    03 --> 04[Task 04: Run Tests & Verify Integration]
```

## Execution Blueprint

**Validation Gates:**
- Reference: `/config/hooks/POST_PHASE.md`

### ✅ Phase 1: Infrastructure Removal
**Parallel Tasks:**
- ✔️ Task 01: Remove obsolete sampling/elicitation/streaming/pending request files and services

### ✅ Phase 2: SDK Integration
**Parallel Tasks:**
- ✔️ Task 02: Create `McpToolSamplingTrait` for tool sampling support (depends on: 01)

### ✅ Phase 3: Gateway Wiring
**Parallel Tasks:**
- ✔️ Task 03: Update `CustomCallToolHandler` and `McpBridgeService` for ClientGateway injection (depends on: 02)

### ✅ Phase 4: Validation
**Parallel Tasks:**
- ✔️ Task 04: Run tests and verify ClientGateway integration (depends on: 03)

### Blueprint Execution Summary
- Total Phases: 4
- Total Tasks: 4
- Maximum Parallelism: 1 task (sequential dependencies)
- Critical Path Length: 4 phases

## Execution Summary

**Status**: ✅ Completed Successfully
**Completed Date**: 2025-11-27

### Results

Successfully migrated from custom sampling/elicitation infrastructure to SDK's ClientGateway:

1. **Task 01 (Infrastructure Removal)**: Deleted ~2,500 lines of obsolete code including 18 files and 5 empty directories (SamplingCoordinator, ElicitationCoordinator, StreamingResult, PendingRequest, etc.)

2. **Task 02 (Sampling Trait)**: Created `McpToolSamplingTrait` at `src/Traits/McpToolSamplingTrait.php` providing `setClientGateway()` and `sample()` methods for tool plugins

3. **Task 03 (Gateway Wiring)**: Updated `CustomCallToolHandler`, `McpBridgeService`, and `ToolApiDiscovery` to pass ClientGateway through the tool execution chain. Tools using the trait receive gateway injection automatically.

4. **Task 04 (Validation)**: All 40 PHPUnit tests pass. PHPStan and PHPCS analysis clean. Gateway injection path verified end-to-end.

### Noteworthy Events

- The SDK uses `Mcp\Server\ClientGateway` (not `Mcp\Client\ClientGateway` as originally documented in tasks)
- Property naming adjusted from `$gateway` to `$clientGateway` to follow Drupal lowerCamelCase conventions
- All deprecation warnings in tests originate from upstream Drupal/dependencies, not mcp_server code

### Recommendations

1. **End-to-end testing**: Verify sampling with MCP Inspector client before release
2. **Documentation**: Update module documentation to explain the `McpToolSamplingTrait` pattern for tool developers
3. **Example tool**: Consider adding an example tool demonstrating sampling capabilities
