---
id: 2
group: "sdk-integration"
dependencies: [1]
status: "completed"
created: 2025-11-27
skills:
  - php
  - drupal
---
# Create McpToolSamplingTrait for Tool Sampling Support

## Objective

Create a PHP trait that provides a clean interface for MCP tools that need sampling capabilities. This trait will allow tools to request LLM sampling via the SDK's `ClientGateway`.

## Skills Required

- PHP 8.3+ traits and interfaces
- MCP SDK (`ClientGateway`, `CreateSamplingMessageResult`)

## Acceptance Criteria

- [x] `McpToolSamplingTrait` created at `src/Traits/McpToolSamplingTrait.php`
- [x] Trait provides `setClientGateway(ClientGateway $gateway): void` method
- [x] Trait provides protected `sample()` method that wraps `ClientGateway::sample()`
- [x] Trait throws `\RuntimeException` if sampling is called without gateway being set
- [x] Code passes PHPStan static analysis
- [x] Code passes PHPCS Drupal/DrupalPractice standards

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

The trait must:
1. Store `ClientGateway` instance via setter injection
2. Provide a protected `sample()` method for tools to use
3. Handle the case where gateway is not set (throw exception)

```php
namespace Drupal\mcp_server\Traits;

use Mcp\Client\ClientGateway;
use Mcp\Schema\Result\CreateSamplingMessageResult;

trait McpToolSamplingTrait {
    protected ?ClientGateway $gateway = NULL;

    public function setClientGateway(ClientGateway $gateway): void {
        $this->gateway = $gateway;
    }

    protected function sample(
        string|array $message,
        int $maxTokens = 1000,
        int $timeout = 120,
        array $options = [],
    ): CreateSamplingMessageResult {
        if ($this->gateway === NULL) {
            throw new \RuntimeException('ClientGateway not available for sampling');
        }
        return $this->gateway->sample($message, $maxTokens, $timeout, $options);
    }
}
```

## Input Dependencies

Task 1: Old sampling infrastructure removed

## Output Artifacts

- `src/Traits/McpToolSamplingTrait.php` - Trait file ready for use by tool plugins

## Implementation Notes

<details>
<summary>Detailed implementation steps</summary>

1. Create directory `src/Traits/` if it doesn't exist
2. Create `McpToolSamplingTrait.php` with:
   - `declare(strict_types=1);`
   - Proper namespace `Drupal\mcp_server\Traits`
   - Import `Mcp\Client\ClientGateway`
   - Import `Mcp\Schema\Result\CreateSamplingMessageResult`
3. The trait provides setter injection pattern - tools using this trait will have `setClientGateway()` called by the handler before execution
4. The `sample()` method signature should match the SDK's `ClientGateway::sample()` parameters
5. Run `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/src/Traits/` to verify
6. Run `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/src/Traits/` to verify

</details>
