---
id: 3
group: "sdk-integration"
dependencies: [2]
status: "completed"
created: 2025-11-27
skills:
  - php
  - drupal
---
# Update CustomCallToolHandler and McpBridgeService for ClientGateway Injection

## Objective

Modify the tool execution flow to create a `ClientGateway` instance from the session and pass it to tools that support sampling via the `McpToolSamplingTrait`.

## Skills Required

- PHP 8.3+ and Drupal dependency injection
- MCP SDK (`ClientGateway`, `SessionInterface`)

## Acceptance Criteria

- [x] `CustomCallToolHandler::handle()` creates `ClientGateway` from session
- [x] `McpBridgeService::executeMcpTool()` accepts optional `?ClientGateway $gateway` parameter
- [x] Gateway is passed to tools that use `McpToolSamplingTrait` via `setClientGateway()`
- [x] Tools without the trait continue to work unchanged
- [x] All existing tests pass
- [x] Code passes PHPStan and PHPCS

Use your internal Todo tool to track these and keep on track.

## Technical Requirements

### CustomCallToolHandler Changes

```php
public function handle(Request $request, SessionInterface $session): Response|Error {
    // Create ClientGateway from session
    $gateway = new ClientGateway($session);

    // Pass gateway to bridge service
    $result = $this->bridge->executeMcpTool($tool_name, $arguments, $gateway);
    // ...
}
```

### McpBridgeService Changes

```php
public function executeMcpTool(
    string $mcpName,
    array $parameters,
    ?ClientGateway $gateway = NULL,  // Add this parameter
): mixed {
    // ... existing authentication logic ...

    return $this->executeToolById($tool_config->getToolId(), $parameters, $gateway);
}

private function executeToolById(
    string $toolId,
    array $parameters,
    ?ClientGateway $gateway = NULL,
): mixed {
    // Get tool instance and inject gateway if it supports sampling
    return $this->toolApiDiscovery->executeTool($toolId, $parameters, $gateway);
}
```

### ToolApiDiscovery Changes

The `executeTool()` method needs to accept the gateway and inject it into tools that use `McpToolSamplingTrait`:

```php
public function executeTool(string $toolId, array $parameters, ?ClientGateway $gateway = NULL): mixed {
    $tool = $this->createToolInstance($toolId);

    // Inject gateway if tool supports sampling
    if ($gateway !== NULL && method_exists($tool, 'setClientGateway')) {
        $tool->setClientGateway($gateway);
    }

    return $tool->execute($parameters);
}
```

## Input Dependencies

Task 2: `McpToolSamplingTrait` created

## Output Artifacts

- Updated `src/Handler/CustomCallToolHandler.php`
- Updated `src/McpBridgeService.php`
- Updated `src/ToolApiDiscovery.php`

## Implementation Notes

<details>
<summary>Detailed implementation steps</summary>

1. **CustomCallToolHandler.php**:
   - Add `use Mcp\Client\ClientGateway;` import
   - In `handle()` method, after getting tool_name and arguments:
     ```php
     $gateway = new ClientGateway($session);
     ```
   - Pass gateway to bridge: `$this->bridge->executeMcpTool($tool_name, $arguments, $gateway)`

2. **McpBridgeService.php**:
   - Add `use Mcp\Client\ClientGateway;` import
   - Add `?ClientGateway $gateway = NULL` parameter to `executeMcpTool()`
   - Add same parameter to `executeToolById()`
   - Pass gateway through to `toolApiDiscovery->executeTool()`

3. **ToolApiDiscovery.php**:
   - Add `use Mcp\Client\ClientGateway;` import
   - Add `?ClientGateway $gateway = NULL` parameter to `executeTool()`
   - Before calling `$tool->execute()`, check if tool has `setClientGateway` method
   - If yes, call `$tool->setClientGateway($gateway)` before execution

4. **Verification**:
   - Run `vendor/bin/drush cache:rebuild`
   - Run `vendor/bin/phpunit web/modules/contrib/mcp_server/tests/`
   - Run `vendor/bin/phpstan analyse web/modules/contrib/mcp_server/`
   - Run `vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/contrib/mcp_server/`

</details>
