---
id: 22
summary: "Implement DynamicToolProviderInterface, DynamicPromptProviderInterface, and DynamicResourceProviderInterface for php/mcp SDK to support runtime-defined MCP elements"
created: 2025-11-30
---

# Plan: Dynamic Provider Interfaces for php/mcp SDK

## Original Work Order

> Implement DynamicToolProviderInterface for the php/mcp SDK to support runtime-defined tools. The implementation should:
>
> 1. Create a new DynamicToolProviderInterface in src/Capability/ that allows:
>    - Listing available tools (getTools(): iterable<Tool>)
>    - Checking if a provider handles a tool (supports(string $toolName): bool)
>    - Executing a tool with arguments (execute(string $toolName, array $arguments, ?ClientGateway $client): mixed)
>
> 2. Modify the RegistryInterface and Registry to:
>    - Store registered dynamic tool providers
>    - Include dynamic tools when listing tools (getTools())
>    - Indicate when a tool is dynamic vs static
>
> 3. Modify CallToolHandler to:
>    - Check dynamic providers first via supports()
>    - Delegate execution to provider's execute() method
>    - Maintain existing ReferenceHandler logic for static tools
>
> 4. Update the Server\Builder to:
>    - Add addDynamicToolProvider() method
>    - Pass providers to the Registry during build()
>
> 5. Add comprehensive tests for:
>    - Provider registration and tool listing
>    - Tool execution through providers
>    - Mixed static and dynamic tools
>    - Provider priority/ordering
>
> The implementation should follow Symfony coding standards and be backward compatible with existing tool registration methods.

## Plan Clarifications

| Question | Answer |
|----------|--------|
| Target repository? | Upstream PR to github.com/modelcontextprotocol/php-sdk |
| Priority when static and dynamic tools have same name? | Error on conflict (throw exception) |
| Should prompts and resources also have dynamic providers? | Yes, include all MCP elements |

## Executive Summary

This plan introduces a provider-based architecture for dynamically loaded MCP elements (tools, prompts, and resources) in the php/mcp SDK. The implementation adds three new interfaces—`DynamicToolProviderInterface`, `DynamicPromptProviderInterface`, and `DynamicResourceProviderInterface`—that allow MCP servers to expose elements whose definitions are determined at runtime rather than through static code or reflection.

The approach chosen separates the concerns of element discovery (listing available elements) from element execution (handling calls). This allows integrators to implement custom providers that source element definitions from databases, configuration files, external APIs, or plugin systems without requiring reflection-based parameter mapping.

This design maintains full backward compatibility with existing registration methods (attributes, closures, manual registration) while adding first-class support for dynamic scenarios common in CMS frameworks, plugin architectures, and gateway/proxy MCP servers.

## Context

### Current State vs Target State

| Current State | Target State | Why? |
|---------------|--------------|------|
| Tools require handlers with reflectable parameters | Tools can be registered with explicit schemas and custom executors | Enables database-driven, plugin-based, and proxy scenarios |
| All tool execution goes through ReferenceHandler | Dynamic tools can bypass ReferenceHandler entirely | Providers control their own execution logic |
| No concept of "provider" for element sources | Registry can aggregate elements from multiple providers | Cleaner separation between element sources |
| Workaround: custom RequestHandler to intercept calls | First-class SDK support for dynamic elements | Reduces boilerplate and improves maintainability |
| Tools only (if workaround used) | Consistent pattern for tools, prompts, and resources | Uniform API across all MCP element types |

### Background

The php/mcp SDK currently supports two primary registration patterns:

1. **Attribute-based discovery**: Methods annotated with `#[McpTool]`, `#[McpResource]`, or `#[McpPrompt]` are discovered via filesystem scanning. The SDK uses reflection to derive input schemas from method parameters.

2. **Manual registration**: The `Builder::addTool()` method accepts a callable handler. The SDK still uses reflection on this handler to derive schemas unless an explicit `inputSchema` is provided.

Both patterns assume the handler is available and introspectable at registration time. This creates friction for:

- **CMS plugins** (Drupal, Symfony bundles, Laravel packages) that define tools via configuration
- **Admin-defined tools** stored in databases
- **Gateway/proxy servers** that aggregate tools from backend services
- **Schema-first development** where tools are defined in JSON/YAML

Current workarounds require implementing custom `RequestHandlerInterface` classes that intercept `CallToolRequest` before the SDK's `CallToolHandler`, duplicating error handling, logging, and result formatting logic.

## Architectural Approach

```mermaid
graph TD
    subgraph "Client"
        A[MCP Client]
    end

    subgraph "Server"
        B[Protocol]
        C[CallToolHandler]
        D[GetPromptHandler]
        E[ReadResourceHandler]
        F[Registry]
        G[ReferenceHandler]
    end

    subgraph "Static Elements"
        H[ToolReference]
        I[PromptReference]
        J[ResourceReference]
    end

    subgraph "Dynamic Providers"
        K[DynamicToolProvider]
        L[DynamicPromptProvider]
        M[DynamicResourceProvider]
    end

    A -->|tools/call| B
    B --> C
    C --> F
    F -->|static tool?| H
    F -->|dynamic tool?| K
    H --> G
    K -->|provider.execute| K

    A -->|prompts/get| B
    B --> D
    D --> F
    F -->|static prompt?| I
    F -->|dynamic prompt?| L

    A -->|resources/read| B
    B --> E
    E --> F
    F -->|static resource?| J
    F -->|dynamic resource?| M
```

### Provider Interfaces

**Objective**: Define contracts for dynamic element providers that decouple element listing from execution.

Three parallel interfaces will be introduced in `src/Capability/`:

- `DynamicToolProviderInterface`: Provides tools with `getTools()`, `supportsTool()`, and `executeTool()`
- `DynamicPromptProviderInterface`: Provides prompts with `getPrompts()`, `supportsPrompt()`, and `getPrompt()`
- `DynamicResourceProviderInterface`: Provides resources with `getResources()`, `supportsResource()`, and `readResource()`

Each interface follows the same pattern:
1. A method to enumerate available elements for discovery
2. A method to check if the provider handles a specific element
3. A method to execute/retrieve the element

The `supports*()` method exists to enable efficient routing without requiring providers to be queried for their full element list on every call.

### Registry Modifications

**Objective**: Extend the Registry to aggregate elements from both static references and dynamic providers.

The `Registry` class will maintain arrays of registered providers alongside existing reference arrays. The `getTools()`, `getPrompts()`, and `getResources()` methods will aggregate elements from both sources.

Name collision detection will be implemented: when a provider returns an element with a name that conflicts with either a static registration or another provider's element, a `RegistryException` will be thrown during registration or first enumeration.

New interface methods will be added:
- `registerDynamicToolProvider(DynamicToolProviderInterface $provider): void`
- `registerDynamicPromptProvider(DynamicPromptProviderInterface $provider): void`
- `registerDynamicResourceProvider(DynamicResourceProviderInterface $provider): void`

The `hasDynamicTool()`, `hasDynamicPrompt()`, and `hasDynamicResource()` methods will check if any provider supports a given element name.

### Handler Modifications

**Objective**: Update request handlers to delegate to dynamic providers when appropriate.

`CallToolHandler` will be modified to:
1. First check if any dynamic provider supports the tool name
2. If yes, delegate to the provider's `executeTool()` method directly
3. If no, fall back to existing `ReferenceHandler` logic for static tools

Similar modifications will be made to `GetPromptHandler` and `ReadResourceHandler`.

The handler will construct a `ClientGateway` from the session and pass it to the provider's execute method, enabling dynamic tools to use sampling/elicitation features.

### Builder Modifications

**Objective**: Provide a fluent API for registering dynamic providers.

New builder methods:
- `addDynamicToolProvider(DynamicToolProviderInterface $provider): self`
- `addDynamicPromptProvider(DynamicPromptProviderInterface $provider): self`
- `addDynamicResourceProvider(DynamicResourceProviderInterface $provider): self`

Providers will be passed to the Registry during `build()`. The builder will also need to handle `hasTools()`, `hasPrompts()`, and `hasResources()` checks that inform server capabilities—these must account for dynamic providers.

### Test Coverage

**Objective**: Ensure comprehensive test coverage for all provider scenarios.

Unit tests will cover:
- Provider registration and deregistration
- Element enumeration from mixed static/dynamic sources
- Conflict detection and exception throwing
- Tool/prompt/resource execution through providers
- Provider ordering (multiple providers, first-match semantics)
- Error handling within provider execution
- Result formatting for dynamic elements

Integration tests will verify the full request flow from protocol through to provider execution.

## Risk Considerations and Mitigation Strategies

<details>
<summary>Technical Risks</summary>

- **Name collision complexity**: Dynamic providers may return different tool sets at different times, making collision detection timing-sensitive
    - **Mitigation**: Validate collisions at registration time and on first enumeration; document that providers should return stable element lists

- **Performance impact on element listing**: Aggregating from multiple providers on every `tools/list` request could be slow
    - **Mitigation**: Providers are expected to cache their element lists internally; SDK can optionally cache the aggregated list

</details>

<details>
<summary>Implementation Risks</summary>

- **Breaking changes to RegistryInterface**: Adding new methods to the interface could break existing implementations
    - **Mitigation**: New methods will have default implementations in the interface using PHP 8's interface default methods, or Registry will remain the only implementation

- **Upstream acceptance**: The php/mcp SDK maintainers may prefer a different approach
    - **Mitigation**: Open an issue first to discuss the design before submitting a PR

</details>

<details>
<summary>Integration Risks</summary>

- **Ordering ambiguity**: When multiple providers support the same tool name, behavior is undefined
    - **Mitigation**: Treat this as a conflict and throw an exception (per clarification)

</details>

## Success Criteria

### Primary Success Criteria

1. DynamicToolProviderInterface, DynamicPromptProviderInterface, and DynamicResourceProviderInterface are implemented and documented
2. Registry correctly aggregates elements from both static and dynamic sources
3. Handlers correctly delegate to providers for dynamic elements
4. Name conflicts between static and dynamic elements throw clear exceptions
5. All new code has corresponding unit tests with >90% coverage
6. Existing tests continue to pass (backward compatibility)
7. PR is accepted by php/mcp SDK maintainers

## Resource Requirements

### Development Skills

- PHP 8.3+ with strict typing and readonly properties
- Symfony coding standards and contribution workflow
- PHPUnit testing practices
- Understanding of MCP protocol and SDK architecture

### Technical Infrastructure

- Fork of github.com/modelcontextprotocol/php-sdk
- PHPUnit 10+ for testing
- PHPStan for static analysis
- PHP-CS-Fixer for code formatting

## Integration Strategy

This feature integrates with existing SDK components:

1. **Registry**: New provider storage alongside existing reference storage
2. **Handlers**: Conditional logic to check providers before references
3. **Builder**: New fluent methods that mirror existing registration patterns
4. **Events**: Existing ToolListChangedEvent/PromptListChangedEvent/ResourceListChangedEvent will be dispatched when providers are registered

The design ensures that servers not using dynamic providers see zero behavioral changes.

## Notes

- The upstream issue #167 discusses custom injection functionality which is related but distinct
- TypeScript MCP SDK has a simpler `server.tool()` pattern; this proposal is more structured to fit PHP's type system
- Consider requesting feedback on the design in a GitHub issue before implementing
