---
id: 23
summary: "Add elicitation support to PHP MCP SDK following the existing sampling architecture pattern"
created: 2025-11-30
---

# Plan: Add Elicitation Support to PHP MCP SDK

## Original Work Order

> Add elicitation support to the PHP MCP SDK (modelcontextprotocol/php-sdk) following the existing sampling architecture. This feature allows servers to request structured user input from clients with JSON Schema validation. Implementation should include: 1) Schema classes (CreateElicitationRequest, CreateElicitationResult, ElicitationAction enum), 2) ClientGateway.elicit() method, 3) ClientAwareTrait extension, 4) ClientCapabilities update, 5) MessageFactory registration, 6) Documentation, 7) Tests. The implementation should mirror the sampling feature for consistency.

## Executive Summary

This plan implements MCP elicitation support in the PHP SDK per the **2025-11-25 specification**, enabling servers to request structured user input from clients during tool execution. The spec defines two modes: **form mode** for JSON Schema-validated input, and **URL mode** for out-of-band authentication flows.

The implementation follows the established sampling architecture pattern in the SDK, ensuring consistency and maintainability. The approach leverages the existing `ClientGateway` infrastructure for request/response handling via PHP Fibers, minimizing new code while maximizing integration with the current codebase.

This feature is essential for building interactive MCP servers that require human input for confirmations, configuration, credentials, or data collection during tool execution.

## Context

### Current State vs Target State

| Current State | Target State | Why? |
|--------------|--------------|------|
| SDK supports `sampling/createMessage` for LLM requests | SDK also supports `elicitation/create` for user input | Enables interactive workflows requiring human data |
| `ClientCapabilities` only tracks `sampling` | `ClientCapabilities` tracks `elicitation` with `form`/`url` sub-capabilities | Servers need granular capability detection |
| `ClientGateway` has `sample()` method only | `ClientGateway` has `elicit()` and `elicitUrl()` methods | Consistent API for both elicitation modes |
| `ClientAwareTrait` exposes only `$this->sample()` | Trait exposes `$this->elicit()` and `$this->elicitUrl()` | Trait users get both capabilities |
| No elicitation notifications | Support for `notifications/elicitation/complete` | URL mode completion signaling |
| Documentation covers sampling only | Documentation covers elicitation (both modes) | Users need guidance on the new feature |

### Background

The MCP specification (2025-11-25) defines two complementary client capabilities:
- **Sampling**: Servers request LLM completions from clients
- **Elicitation**: Servers request structured user input from clients

The elicitation specification defines **two modes**:

1. **Form Mode** (`mode: "form"` or omitted):
   - Request params: `message` (string), `requestedSchema` (JSON Schema)
   - Response: `action` (accept/decline/cancel) and `content` (matching schema, on accept)
   - Schema restrictions: flat objects with primitive properties only

2. **URL Mode** (`mode: "url"`):
   - Request params: `message` (string), `url` (string), `elicitationId` (string)
   - Response: `action` (accept/decline/cancel) — no content field
   - Used for sensitive data collection via out-of-band web flows
   - Server sends `notifications/elicitation/complete` when flow completes

**Capability Declaration** (per spec):
```json
{
  "capabilities": {
    "elicitation": {
      "form": {},
      "url": {}
    }
  }
}
```

## Architectural Approach

```mermaid
graph TB
    subgraph "Schema Layer"
        A[CreateElicitationRequest] --> B[Request base class]
        C[CreateElicitationResult] --> D[ResultInterface]
        E[ElicitationAction enum]
        F[ElicitationMode enum]
        G[ElicitationCompleteNotification] --> H[Notification base]
    end

    subgraph "Server Layer"
        I[ClientGateway] --> J[elicit method - form]
        I --> K[elicitUrl method - url]
        L[ClientAwareTrait] --> M[elicit wrapper]
        L --> N[elicitUrl wrapper]
    end

    subgraph "Infrastructure Layer"
        O[ClientCapabilities] --> P[elicitation object]
        P --> Q[form capability]
        P --> R[url capability]
        S[MessageFactory] --> T[request registration]
    end

    J --> A
    K --> A
    J --> C
    K --> C
```

### Schema Classes

**Objective**: Define the request, result, enum, and notification types for elicitation following existing SDK patterns.

**New Classes:**

1. **`ElicitationMode`** enum (`src/Schema/Enum/`):
   - `FORM = 'form'`
   - `URL = 'url'`

2. **`ElicitationAction`** enum (`src/Schema/Enum/`):
   - `ACCEPT = 'accept'`
   - `DECLINE = 'decline'`
   - `CANCEL = 'cancel'`

3. **`CreateElicitationRequest`** (`src/Schema/Request/`):
   - Properties: `message`, `mode`, `requestedSchema` (form), `url` (url), `elicitationId` (url)
   - Method: `elicitation/create`
   - Validates mode-specific required parameters

4. **`CreateElicitationResult`** (`src/Schema/Result/`):
   - Properties: `action` (ElicitationAction), `content` (optional array)
   - `content` only present when `action === ACCEPT` in form mode

5. **`ElicitationCompleteNotification`** (`src/Schema/Notification/`):
   - Properties: `elicitationId`
   - Method: `notifications/elicitation/complete`
   - Sent by server when URL mode flow completes

### ClientGateway Extension

**Objective**: Add convenience methods to `ClientGateway` for both elicitation modes.

**Form Mode - `elicit()`:**
```php
public function elicit(
    string $message,
    array $requestedSchema,
    int $timeout = 120
): CreateElicitationResult
```
- Sends form mode request
- Returns typed result with `action` and optional `content`
- Throws `ClientException` on error

**URL Mode - `elicitUrl()`:**
```php
public function elicitUrl(
    string $message,
    string $url,
    string $elicitationId,
    int $timeout = 120
): CreateElicitationResult
```
- Sends URL mode request
- Returns typed result with `action` only
- Throws `ClientException` on error

Location: `src/Server/ClientGateway.php`

### ClientAwareTrait Extension

**Objective**: Expose elicitation to trait users with the same pattern as sampling.

Add protected methods:
- `elicit()` - delegates to `$this->client->elicit()`
- `elicitUrl()` - delegates to `$this->client->elicitUrl()`

Location: `src/Server/ClientAwareTrait.php`

### ClientCapabilities Update

**Objective**: Track client elicitation capabilities with granular mode support.

The spec defines elicitation capability as an object with optional `form` and `url` sub-objects:

```php
public readonly ?ElicitationCapabilities $elicitation = null;
```

Where `ElicitationCapabilities` is a new class:
```php
final class ElicitationCapabilities {
    public function __construct(
        public readonly bool $form = false,
        public readonly bool $url = false,
    ) {}
}
```

Location: `src/Schema/ClientCapabilities.php`, `src/Schema/ElicitationCapabilities.php`

### MessageFactory Registration

**Objective**: Register the new request and notification classes for message parsing.

Add to `REGISTERED_MESSAGES`:
- `CreateElicitationRequest::class`
- `ElicitationCompleteNotification::class`

Location: `src/JsonRpc/MessageFactory.php`

### Documentation

**Objective**: Document the elicitation feature in the client communication guide.

Add an "Elicitation" section to `docs/client-communication.md` covering:
- Form mode usage with schema examples
- URL mode usage for sensitive data
- Capability detection
- Action handling (accept/decline/cancel)
- Security considerations

Location: `docs/client-communication.md`

### Unit Tests

**Objective**: Verify the schema classes work correctly.

Create tests for:
- `CreateElicitationRequest` - form mode construction and validation
- `CreateElicitationRequest` - URL mode construction and validation
- `CreateElicitationResult` - deserialization with all action types
- `ElicitationAction` and `ElicitationMode` enum values
- `ElicitationCompleteNotification` construction
- `ElicitationCapabilities` parsing

Location: `tests/Unit/Schema/`

## Risk Considerations and Mitigation Strategies

<details>
<summary>Technical Risks</summary>

- **Schema validation complexity**: The MCP spec restricts schemas to flat objects with primitives only
    - **Mitigation**: Document limitations clearly; validation is the client's responsibility per spec

- **Mode parameter handling**: Request must validate mode-specific required parameters
    - **Mitigation**: Throw `InvalidArgumentException` in constructor if URL mode missing `url`/`elicitationId`
</details>

<details>
<summary>Implementation Risks</summary>

- **Fiber handling edge cases**: The elicit methods use the same Fiber suspension pattern as sample()
    - **Mitigation**: Reuse the existing `request()` private method which handles Fiber correctly
</details>

<details>
<summary>Integration Risks</summary>

- **Breaking changes to existing code**: Adding new parameters to ClientCapabilities
    - **Mitigation**: Use nullable optional parameters with defaults to maintain BC

- **Capability structure change**: `elicitation` is now an object, not boolean
    - **Mitigation**: Handle both formats in `fromArray()` for forward compatibility
</details>

## Success Criteria

### Primary Success Criteria

1. Server tools can request user input via `$client->elicit()` (form mode) and receive structured responses
2. Server tools can initiate URL flows via `$client->elicitUrl()` for sensitive data collection
3. Client elicitation capability is detected with granular form/url support via `ClientCapabilities->elicitation`
4. `notifications/elicitation/complete` can be sent when URL flows complete
5. All unit tests pass for the new schema classes
6. Documentation clearly explains both elicitation modes with examples
7. Implementation follows established SDK patterns (code review passes)

## Resource Requirements

### Development Skills

- PHP 8.1+ with readonly properties and enums
- Understanding of JSON-RPC message patterns
- Familiarity with the MCP specification (2025-11-25)

### Technical Infrastructure

- PHP SDK repository (modelcontextprotocol/php-sdk)
- PHPUnit for testing
- PHPStan for static analysis

## Integration Strategy

The implementation integrates with existing SDK infrastructure:
- Uses `ClientGateway`'s existing `request()` method for Fiber-based client communication
- Follows the same pattern as `CreateSamplingMessageRequest`/`CreateSamplingMessageResult`
- Registers in `MessageFactory` alongside other request types
- Documented alongside sampling in `client-communication.md`

## Notes

- The MCP elicitation specification intentionally limits form schemas to flat objects with primitive properties to simplify client implementation
- Response actions (`accept`, `decline`, `cancel`) differ from sampling's single response pattern - callers must handle all three cases
- **Form mode**: Servers must NOT request sensitive information (use URL mode instead)
- **URL mode**: Servers must NOT include user credentials in URLs; clients must NOT auto-fetch or pre-authenticate URLs
- Error code `-32042` (`URLElicitationRequiredError`) indicates URL mode is required before retrying an operation
