# MCP Server for Drupal

Drupal integration module for the [Model Context Protocol (MCP)](https://modelcontextprotocol.io/) using the official [mcp/sdk](https://github.com/modelcontextprotocol/php-sdk).

This module enables AI assistants (like Claude, ChatGPT, etc.) to interact with your Drupal site through MCP tools, resources, and prompts. It provides both STDIO (command-line) and HTTP transports for flexible integration.

## Table of Contents

- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Usage](#usage)
  - [STDIO Transport (Command-line)](#stdio-transport-command-line)
  - [HTTP Transport (Web-based)](#http-transport-web-based)
- [Creating MCP Tools](#creating-mcp-tools)
- [Troubleshooting](#troubleshooting)
- [Resources](#resources)
- [Contributing](#contributing)
- [License](#license)
- [Credits](#credits)

## Features

- **Dynamic Tool Discovery**: MCP tools are provided dynamically from configuration entities
- **Tool API Integration**: Expose any Tool API tool as an MCP tool without code changes
- **Dual Transport Support**:
  - STDIO transport for command-line integration (via Drush)
  - HTTP transport for web-based communication
- **Admin UI**: Full CRUD interface for managing MCP tool configurations
- **Drupal Service Integration**: Leverages Drupal's dependency injection container
- **Session Management**: File-based session storage for MCP server state
- **Configurable**: Server name, version, pagination, and session TTL via Drupal config

## Requirements

- Drupal 10.x or 11.x
- PHP 8.3 or higher
- Composer
- Drush 12.x or higher (for STDIO transport)

## Installation

1. Install the module via Composer:

```bash
composer require drupal/mcp_server
```

2. Enable the module:

```bash
drush pm:enable mcp_server
```

3. Clear cache:

```bash
drush cache:rebuild
```

## Usage

### STDIO Transport (Command-line)

The STDIO transport allows AI assistants to communicate with your Drupal site via standard input/output streams.

#### Configuration Example (Claude Desktop)

Add to your MCP client configuration (e.g., `~/Library/Application Support/Claude/claude_desktop_config.json`):

```json
{
  "mcpServers": {
    "drupal": {
      "command": "vendor/bin/drush",
      "args": ["mcp:server"],
      "cwd": "/path/to/your/drupal/site"
    }
  }
}
```

#### Manual Testing

You can test the STDIO server manually:

```bash
drush mcp:server
```

Or use the MCP Inspector:

```bash
npx @modelcontextprotocol/inspector vendor/bin/drush mcp:server
```

### HTTP Transport (Web-based)

The HTTP transport exposes an endpoint for MCP communication via HTTP POST requests.

#### Endpoint

```
POST https://your-drupal-site.com/_mcp
```

#### Example Configuration

For web-based MCP clients, configure them to use:

```json
{
  "mcpServers": {
    "drupal-http": {
      "url": "https://your-drupal-site.com/_mcp",
      "transport": "http"
    }
  }
}
```

## Creating MCP Tools

The MCP Server module integrates with the [Tool API module](https://www.drupal.org/project/tool) to expose any Tool API tool as an MCP tool without requiring code modifications. The Tool API module is a required dependency and will be installed automatically with mcp_server.

**Important**: The mcp_server module uses dynamic discovery based on configuration entities. Tools are NOT created using PHP attributes. Instead, all MCP tools come from the Tool API module and are configured through the admin UI.

### Workflow

1. **Create a Tool API Tool** in your custom module (see [Tool API documentation](https://www.drupal.org/project/tool))
2. **Navigate to the admin UI** at `/admin/config/services/mcp-server/tools`
3. **Add an MCP Tool Configuration**:
   - Enter a label and machine name
   - Select a Tool API tool ID (autocomplete will suggest available tools)
   - Optionally override the MCP tool name and description
   - Ensure "Enabled" is checked
   - Save
4. **The tool is now available** to AI assistants via MCP protocol

### Authentication Modes

Each MCP tool configuration supports two authentication modes:

- **`required`**: OAuth2 Bearer token must be present with all specified scopes. Requests without valid authentication receive HTTP 401. Requests with insufficient scopes receive HTTP 403.
- **`disabled`**: No authentication checks performed. Tool executes for all requests.

Use `required` for tools that modify data or access sensitive information. Use `disabled` only for public, read-only tools with no security concerns.

### Example: Exposing a Custom Tool

<details>
<summary>Click to expand example code</summary>

If you have a Tool API tool defined in your custom module:

```php
// In my_module/src/Plugin/Tool/SendEmail.php
#[Tool(
  id: 'my_module:send_email',
  label: 'Send Email',
  description: 'Sends an email to a specified recipient'
)]
class SendEmail {

  public function buildInputDefinitions(): array {
    return [
      'to' => InputDefinition::create('email')
        ->setLabel('Recipient')
        ->setDescription('Email address of the recipient')
        ->setRequired(TRUE),
      'subject' => InputDefinition::create('string')
        ->setLabel('Subject')
        ->setRequired(TRUE),
      'body' => InputDefinition::create('string')
        ->setLabel('Body')
        ->setRequired(TRUE),
    ];
  }

  protected function doExecute(): void {
    $to = $this->getInputValue('to');
    $subject = $this->getInputValue('subject');
    $body = $this->getInputValue('body');

    // Send email logic...

    $this->setResultStatus(TRUE);
    $this->setResultMessage('Email sent successfully');
  }
}
```

Create an MCP configuration to expose it:

1. Admin UI: Add config with tool_id = `my_module:send_email`
2. The tool will now be available to MCP clients as an MCP tool
3. AI assistants can discover and call it via the MCP protocol

</details>

### Schema Conversion

Tool API's InputDefinition types are automatically converted to MCP JSON Schema:

- `string` → `string`
- `integer` → `integer`
- `float` → `number`
- `boolean` → `boolean`
- `email` → `string`
- `uri` → `string`
- `entity:*` / `entity_reference:*` → `string` (entity ID)
- `list` → `array`
- `map` → `object`

Constraints are also converted:

- Length constraints → `minLength`, `maxLength`
- Range constraints → `minimum`, `maximum`
- Regex patterns → `pattern`
- Allowed values → `enum`

## Troubleshooting

### Check if the module is enabled

```bash
drush pm:list --type=module --status=enabled | grep mcp_server
```

### View MCP server logs

```bash
drush watchdog:show --filter=mcp_server
```

### Test MCP capabilities discovery

```bash
drush eval "print_r(\Drupal::service('mcp_server.server'));"
```

### Clear cache after adding new MCP capabilities

```bash
drush cache:rebuild
```

## Resources

- [Model Context Protocol Documentation](https://modelcontextprotocol.io)
- [MCP PHP SDK](https://github.com/modelcontextprotocol/php-sdk)
- [MCP Specification](https://spec.modelcontextprotocol.io)
- [Symfony MCP Bundle](https://github.com/symfony/mcp-bundle)

## Contributing

Contributions are welcome! Please submit issues and pull requests on the module's issue queue.

## License

This project is licensed under GPL-2.0-or-later.

## Credits

This module integrates the official [MCP PHP SDK](https://github.com/modelcontextprotocol/php-sdk), which is a collaboration between the PHP Foundation and the Symfony project.

Inspired by the [Symfony MCP Bundle](https://github.com/symfony/mcp-bundle).
