<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Capability\Loader;

use Drupal\mcp_server\McpBridgeService;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;
use Mcp\Capability\Registry\Loader\LoaderInterface;
use Mcp\Capability\Registry\ReferenceRegistryInterface;
use Mcp\Schema\Prompt;
use Mcp\Schema\PromptArgument;
use Psr\Log\LoggerInterface;

/**
 * Custom loader for registering prompts from Drupal configuration.
 *
 * This loader bypasses the SDK's reflection-based argument discovery and
 * creates Prompt objects directly from configuration, properly exposing
 * argument descriptions and metadata.
 */
final class PromptConfigLoader implements LoaderInterface {

  /**
   * Constructs a PromptConfigLoader.
   *
   * @param \Drupal\mcp_server\McpBridgeService $mcpBridge
   *   The MCP bridge service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager $completionProviderManager
   *   The completion provider plugin manager.
   */
  public function __construct(
    private readonly McpBridgeService $mcpBridge,
    private readonly LoggerInterface $logger,
    private readonly PromptArgumentCompletionProviderManager $completionProviderManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function load(ReferenceRegistryInterface $registry): void {
    $enabled_prompts = $this->mcpBridge->getEnabledPrompts();

    foreach ($enabled_prompts as $prompt_data) {
      try {
        // Create PromptArgument objects from configuration.
        $arguments = [];
        foreach ($prompt_data['arguments'] ?? [] as $arg) {
          $arguments[] = new PromptArgument(
            name: $arg['machine_name'],
            title: $arg['label'],
            description: $arg['description'] ?? NULL,
            required: $arg['required'] ?? FALSE,
          );
        }

        // Create Prompt object with all metadata from configuration.
        $prompt = new Prompt(
          name: $prompt_data['name'],
          description: $prompt_data['description'] ?? NULL,
          arguments: $arguments,
          icons: NULL,
          meta: $prompt_data['title'] ? ['title' => $prompt_data['title']] : NULL,
        );

        // Handler that returns messages from configuration.
        $handler = fn(array $args = []) => $prompt_data['messages'] ?? [];

        // Build completion providers array.
        $completion_providers = $this->buildCompletionProviders($prompt_data);

        // Register prompt with the registry.
        $registry->registerPrompt($prompt, $handler, $completion_providers, TRUE);

        $this->logger->debug(
          "Registered prompt {$prompt_data['name']} from configuration"
        );
        if (!empty($completion_providers)) {
          $this->logger->debug(
            'Registered @count completion providers for prompt "@prompt"',
            [
              '@count' => count($completion_providers),
              '@prompt' => $prompt_data['name'],
            ]
          );
        }
      }
      catch (\Throwable $e) {
        $this->logger->error(
          'Failed to register prompt @name: @message',
          [
            '@name' => $prompt_data['name'],
            '@message' => $e->getMessage(),
          ]
        );
      }
    }
  }

  /**
   * Builds completion providers for prompt arguments.
   *
   * @param array $prompt_data
   *   The prompt configuration data.
   *
   * @return array
   *   Array mapping argument names to completion provider instances.
   */
  private function buildCompletionProviders(array $prompt_data): array {
    $completion_providers = [];

    foreach ($prompt_data['arguments'] ?? [] as $arg) {
      if (!empty($arg['completion_providers'])) {
        $providers_for_arg = [];

        foreach ($arg['completion_providers'] as $provider_config) {
          try {
            $plugin = $this->completionProviderManager->createInstance(
              $provider_config['plugin_id'],
              $provider_config['configuration'] ?? []
            );

            // Wrap in adapter.
            $providers_for_arg[] = new PluginProviderAdapter(
              $plugin,
              $provider_config['configuration'] ?? []
            );

          }
          catch (\Throwable $e) {
            $this->logger->warning(
              'Failed to instantiate completion provider "@plugin_id" for argument "@arg": @message',
              [
                '@plugin_id' => $provider_config['plugin_id'],
                '@arg' => $arg['machine_name'],
                '@message' => $e->getMessage(),
              ]
            );
          }
        }

        if (!empty($providers_for_arg)) {
          $completion_providers[$arg['machine_name']] = count($providers_for_arg) === 1
            ? $providers_for_arg[0]
            : new ChainedCompletionProvider($providers_for_arg);
        }
      }
    }

    return $completion_providers;
  }

}
