<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Commands;

use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Drupal\mcp_server\Exception\InsufficientScopeException;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Mcp\Server;
use Mcp\Server\Transport\StdioTransport;

/**
 * Drush commands for MCP Server.
 *
 * Provides STDIO transport for MCP server, allowing AI assistants to interact
 * with Drupal via standard input/output streams.
 */
final class McpServerCommands extends DrushCommands {

  /**
   * Constructs a new McpServerCommands object.
   */
  public function __construct(
    private readonly Server $mcpServer,
  ) {
    parent::__construct();
  }

  /**
   * Start MCP server with STDIO transport.
   *
   * This command starts an MCP server that communicates via standard
   * input/output. It's designed to be used with AI assistant clients like
   * Claude Desktop.
   *
   * Example usage in MCP client configuration (e.g., Claude Desktop):
   * @code
   * {
   *   "mcpServers": {
   *     "drupal": {
   *       "command": "vendor/bin/drush",
   *       "args": ["mcp:server"]
   *     }
   *   }
   * }
   * @endcode
   */
  #[CLI\Command(name: 'mcp:server', aliases: ['mcps'])]
  #[CLI\Usage(name: 'drush mcp:server', description: 'Start the MCP server with STDIO transport')]
  public function server(): void {
    try {
      $this->logger()->info('Starting MCP server with STDIO transport...');

      $transport = new StdioTransport();
      $this->mcpServer->run($transport);

      $this->logger()->info('MCP server stopped.');
    }
    catch (AuthenticationRequiredException $e) {
      // Authentication required for tool execution (STDIO transport).
      // Error code -32001: Authentication required (PRD 1).
      $this->logger()->info('Authentication required for tool @tool', [
        '@tool' => $e->getToolName(),
      ]);

      // Send JSON-RPC error response to STDIO.
      $error_response = json_encode([
        'jsonrpc' => '2.0',
        'error' => [
          'code' => -32001,
          'message' => 'Authentication required',
          'data' => [
            'tool' => $e->getToolName(),
            'authentication_mode' => $e->getAuthenticationMode(),
          ],
        ],
        'id' => NULL,
      ]);

      fwrite(STDOUT, $error_response . "\n");
      throw $e;
    }
    catch (InsufficientScopeException $e) {
      // Insufficient OAuth scopes for tool execution (STDIO transport).
      // Error code -32003: Insufficient scope (PRD 2).
      $this->logger()->info('Insufficient scopes for tool @tool', [
        '@tool' => $e->getErrorData()['tool'] ?? '',
      ]);

      // Send JSON-RPC error response to STDIO.
      $error_response = json_encode([
        'jsonrpc' => '2.0',
        'error' => [
          'code' => -32003,
          'message' => 'Insufficient OAuth scopes',
          'data' => $e->getErrorData(),
        ],
        'id' => NULL,
      ]);

      fwrite(STDOUT, $error_response . "\n");
      throw $e;
    }
    catch (\Exception $e) {
      $this->logger()->error('MCP server error: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

}
