<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Drupal\mcp_server\Exception\InsufficientScopeException;
use Mcp\Server;
use Mcp\Server\Transport\StreamableHttpTransport;
use Psr\Http\Message\ResponseFactoryInterface;
use Psr\Http\Message\StreamFactoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Bridge\PsrHttpMessage\HttpFoundationFactoryInterface;
use Symfony\Bridge\PsrHttpMessage\HttpMessageFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for MCP Server HTTP transport.
 *
 * Provides an HTTP endpoint for MCP server, allowing AI assistants to interact
 * with Drupal via HTTP requests.
 */
final class McpServerController extends ControllerBase {

  /**
   * Constructs a new McpServerController object.
   */
  public function __construct(
    private readonly Server $mcpServer,
    private readonly HttpMessageFactoryInterface $httpMessageFactory,
    private readonly HttpFoundationFactoryInterface $httpFoundationFactory,
    private readonly ResponseFactoryInterface $responseFactory,
    private readonly StreamFactoryInterface $streamFactory,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('mcp_server.server'),
      $container->get('psr7.http_message_factory'),
      $container->get('psr7.http_foundation_factory'),
      $container->get('mcp_server.psr17_factory'),
      $container->get('mcp_server.psr17_factory'),
      $container->get('logger.channel.mcp_server'),
    );
  }

  /**
   * Handle MCP server HTTP requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response.
   */
  public function handle(Request $request): Response {
    $this->logger->info('MCP request received');

    $this->logger->info('User is: @user', [
      '@user' => $this->currentUser()->isAnonymous() ? 'anonymous' : $this->currentUser()->getAccountName(),
    ]);

    try {
      $psrRequest = $this->httpMessageFactory->createRequest($request);

      // Standard MCP request (tool call, resource access, etc.).
      $transport = new StreamableHttpTransport(
        $psrRequest,
        $this->responseFactory,
        $this->streamFactory,
        logger: $this->logger,
      );

      $psr_response = $this->mcpServer->run($transport);

      // Log response status and body length.
      $this->logger->info('Response status: @status, Body length: @length', [
        '@status' => $psr_response->getStatusCode(),
        '@length' => $psr_response->getBody()->getSize(),
      ]);

      // Check if the response contains an authentication error.
      // The MCP Server library catches exceptions from tool handlers and
      // converts them to JSON-RPC INTERNAL_ERROR (-32603). We detect our
      // custom error types by checking for special markers in the message.
      $body = $psr_response->getBody();

      // Skip error inspection for streaming SSE responses (CallbackStream).
      // These are not seekable and don't contain JSON error payloads.
      // SSE responses must use streamed mode for proper incremental delivery.
      if (!$body->isSeekable()) {
        return $this->httpFoundationFactory->createResponse($psr_response, TRUE);
      }

      $response_body = (string) $body;
      $json_response = json_decode($response_body, TRUE);

      // Rewind the stream so it can be read again later.
      $body->rewind();

      if (isset($json_response['error']) && is_array($json_response['error'])) {
        $error_message = $json_response['error']['message'] ?? '';

        // Check for authentication error marker.
        if (str_contains($error_message, '[AUTH_REQUIRED]')) {
          $this->logger->info('Converting MCP authentication error to HTTP 401');

          // Extract request ID and data from the error response.
          $request_id = $json_response['id'] ?? NULL;
          $error_data = $json_response['error']['data'] ?? [];

          $response = new JsonResponse([
            'jsonrpc' => '2.0',
            'error' => [
              'code' => -32001,
              'message' => 'Authentication required',
              'data' => $error_data,
            ],
            'id' => $request_id,
          ], 401);

          $response->headers->set('WWW-Authenticate', 'Bearer realm="mcp_server"');
          return $response;
        }

        // Check for insufficient scope error marker.
        if (str_contains($error_message, '[INSUFFICIENT_SCOPE]')) {
          $this->logger->info('Converting MCP scope error to HTTP 403');

          // Extract request ID and data from the error response.
          $request_id = $json_response['id'] ?? NULL;
          $error_data = $json_response['error']['data'] ?? [];

          $response = new JsonResponse([
            'jsonrpc' => '2.0',
            'error' => [
              'code' => -32003,
              'message' => 'Insufficient OAuth scopes',
              'data' => $error_data,
            ],
            'id' => $request_id,
          ], 403);

          $response->headers->set('WWW-Authenticate', 'Bearer error="insufficient_scope"');
          return $response;
        }
      }

      return $this->httpFoundationFactory->createResponse($psr_response);
    }
    catch (AuthenticationRequiredException $e) {
      // Authentication required for tool execution.
      // Error code -32001: Authentication required (PRD 1).
      // Error code -32002: Reserved for authorization failures (PRD 2).
      $this->logger->info('Authentication required for tool @tool', [
        '@tool' => $e->getToolName(),
      ]);

      // Extract request ID from request if available.
      $request_data = json_decode($request->getContent(), TRUE);
      $request_id = $request_data['id'] ?? NULL;

      $response = new JsonResponse([
        'jsonrpc' => '2.0',
        'error' => [
          'code' => -32001,
          'message' => 'Authentication required',
          'data' => [
            'tool' => $e->getToolName(),
            'authentication_mode' => $e->getAuthenticationMode(),
          ],
        ],
        'id' => $request_id,
      ], 401);

      $response->headers->set('WWW-Authenticate', 'Bearer realm="mcp_server"');

      return $response;
    }
    catch (InsufficientScopeException $e) {
      // Insufficient OAuth scopes for tool execution.
      // Error code -32003: Insufficient scope (PRD 2).
      $this->logger->info('Insufficient scopes for tool @tool', [
        '@tool' => $e->getErrorData()['tool'] ?? '',
      ]);

      // Extract request ID from request if available.
      $request_data = json_decode($request->getContent(), TRUE);
      $request_id = $request_data['id'] ?? NULL;

      $response = new JsonResponse([
        'jsonrpc' => '2.0',
        'error' => [
          'code' => -32003,
          'message' => 'Insufficient OAuth scopes',
          'data' => $e->getErrorData(),
        ],
        'id' => $request_id,
      ], 403);

      $response->headers->set(
        'WWW-Authenticate',
        'Bearer error="insufficient_scope"'
      );

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('MCP server HTTP error: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

}
