<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\mcp_server\ToolApiDiscovery;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides autocomplete suggestions for Tool API tools.
 */
final class ToolAutocompleteController extends ControllerBase {

  /**
   * Constructs a ToolAutocompleteController.
   */
  public function __construct(
    private readonly ToolApiDiscovery $toolApiDiscovery,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('mcp_server.tool_api_discovery')
    );
  }

  /**
   * Handles autocomplete requests for Tool API tool IDs.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing the search query.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with matching tool suggestions.
   */
  public function handleAutocomplete(Request $request): JsonResponse {
    $matches = [];
    $query = $request->query->get('q');

    if (!$query) {
      return new JsonResponse($matches);
    }

    $query = strtolower($query);
    $allTools = $this->toolApiDiscovery->getAvailableTools();

    foreach ($allTools as $toolId => $toolDef) {
      // Match against tool ID or label.
      $label = $toolDef['label'] ?? $toolId;
      $searchableText = strtolower($toolId . ' ' . $label);

      if (str_contains($searchableText, $query)) {
        $matches[] = [
          'value' => $toolId,
          'label' => sprintf('%s (%s)', $label, $toolId),
        ];

        // Limit results to prevent overwhelming the UI.
        if (count($matches) >= 10) {
          break;
        }
      }
    }

    return new JsonResponse($matches);
  }

}
