<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Entity;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;

/**
 * Defines the MCP Prompt Configuration entity.
 *
 * This configuration entity stores MCP prompt metadata including name, title,
 * description, arguments, and messages with support for text, images, audio,
 * and embedded resources.
 *
 * @ConfigEntityType(
 *   id = "mcp_prompt_config",
 *   label = @Translation("MCP Prompt Configuration"),
 *   label_collection = @Translation("MCP Prompt Configurations"),
 *   label_singular = @Translation("MCP prompt configuration"),
 *   label_plural = @Translation("MCP prompt configurations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count MCP prompt configuration",
 *     plural = "@count MCP prompt configurations",
 *   ),
 *   handlers = {
 *     "storage" = "Drupal\Core\Config\Entity\ConfigEntityStorage",
 *     "list_builder" = "Drupal\mcp_server\McpPromptConfigListBuilder",
 *     "form" = {
 *       "add" = "Drupal\mcp_server\Form\McpPromptConfigForm",
 *       "edit" = "Drupal\mcp_server\Form\McpPromptConfigForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     },
 *   },
 *   config_prefix = "mcp_prompt_config",
 *   admin_permission = "administer mcp prompt configurations",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "title",
 *     "description",
 *     "arguments",
 *     "messages",
 *     "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/config/services/mcp-server/prompt/{mcp_prompt_config}",
 *     "add-form" = "/admin/config/services/mcp-server/prompt/add",
 *     "edit-form" = "/admin/config/services/mcp-server/prompt/{mcp_prompt_config}/edit",
 *     "delete-form" = "/admin/config/services/mcp-server/prompt/{mcp_prompt_config}/delete",
 *     "collection" = "/admin/config/services/mcp-server/prompts",
 *   }
 * )
 */
final class McpPromptConfig extends ConfigEntityBase {

  /**
   * The prompt title (optional).
   */
  protected ?string $title = NULL;

  /**
   * The prompt description (optional).
   */
  protected ?string $description = NULL;

  /**
   * Array of argument definitions.
   *
   * Each argument has: label (string), machine_name (string),
   * description (string), required (bool), completion_providers (array,
   * optional).
   *
   * Completion providers structure:
   * - plugin_id (string): The completion provider plugin ID
   * - configuration (array, optional): Plugin-specific configuration
   *
   * @var array<int, array{
   *   label: string,
   *   machine_name: string,
   *   description: string,
   *   required: bool,
   *   completion_providers?: array<int, array{
   *     plugin_id: string,
   *     configuration?: array
   *   }>
   * }>
   */
  protected array $arguments = [];

  /**
   * Array of message objects with role and typed content.
   *
   * Each message has role ('user' or 'assistant') and content array.
   * Content items support text, image, audio, and resource types.
   *
   * Content types:
   * - text: {type: 'text', text: 'content'}
   * - image: {type: 'image', data: 'base64...', mimeType: 'image/png'}
   * - audio: {type: 'audio', data: 'base64...', mimeType: 'audio/wav'}
   * - resource: {type: 'resource', resource: {uri: '...', mimeType: '...',
   *   text: '...'}}.
   *
   * @var array<int, array{role: string, content: array}>
   */
  protected array $messages = [];

  /**
   * Gets the prompt title.
   */
  public function getTitle(): ?string {
    return $this->title;
  }

  /**
   * Gets the prompt description.
   */
  public function getDescription(): ?string {
    return $this->description;
  }

  /**
   * Gets the argument definitions.
   *
   * @return array<int, array{label: string, machine_name: string, description: string, required: bool}>
   *   Array of argument definitions.
   */
  public function getArguments(): array {
    return $this->arguments;
  }

  /**
   * Gets the message objects.
   *
   * @return array<int, array{role: string, content: array}>
   *   Array of messages with role and content.
   */
  public function getMessages(): array {
    return $this->messages;
  }

  /**
   * Gets completion providers for a specific argument.
   *
   * @param string $argument_name
   *   The argument machine name.
   *
   * @return array
   *   Array of completion provider configurations.
   */
  public function getArgumentCompletionProviders(string $argument_name): array {
    foreach ($this->arguments as $argument) {
      if ($argument['machine_name'] === $argument_name) {
        return $argument['completion_providers'] ?? [];
      }
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    if (empty($this->id())) {
      throw new \InvalidArgumentException('The id field is required.');
    }

    if (empty($this->label())) {
      throw new \InvalidArgumentException('The label field is required.');
    }

    $this->validateArguments();
    $this->validateMessages();
  }

  /**
   * Validates argument definitions.
   *
   * @throws \InvalidArgumentException
   *   When argument structure is invalid.
   */
  private function validateArguments(): void {
    foreach ($this->arguments as $index => $argument) {
      // PHPDoc guarantees array type, check for required structure.
      if (empty($argument['machine_name'])) {
        throw new \InvalidArgumentException(sprintf(
          'Argument at index %d must have a valid machine_name.',
          $index
        ));
      }

      // Validate completion providers if present.
      if (isset($argument['completion_providers'])) {
        $this->validateCompletionProviders($argument, $index);
      }
    }
  }

  /**
   * Validates completion provider configuration for an argument.
   *
   * @param array $argument
   *   The argument configuration.
   * @param int $index
   *   The argument index for error reporting.
   *
   * @throws \InvalidArgumentException
   *   When completion provider structure is invalid.
   */
  private function validateCompletionProviders(array $argument, int $index): void {
    if (!is_array($argument['completion_providers'])) {
      throw new \InvalidArgumentException(
        sprintf('Argument "%s" completion_providers must be an array.', $argument['machine_name'])
      );
    }

    $plugin_manager = $this->getCompletionProviderManager();
    $available_plugins = array_keys($plugin_manager->getDefinitions());

    foreach ($argument['completion_providers'] as $provider_index => $provider) {
      // Validate plugin_id exists.
      if (empty($provider['plugin_id']) || !is_string($provider['plugin_id'])) {
        throw new \InvalidArgumentException(
          sprintf('Argument "%s" completion provider at index %d must have a valid plugin_id.',
            $argument['machine_name'], $provider_index)
        );
      }

      // Validate plugin exists.
      if (!in_array($provider['plugin_id'], $available_plugins, TRUE)) {
        throw new \InvalidArgumentException(
          sprintf('Argument "%s" references unknown completion provider plugin "%s".',
            $argument['machine_name'], $provider['plugin_id'])
        );
      }

      // Validate configuration is array if present.
      if (isset($provider['configuration']) && !is_array($provider['configuration'])) {
        throw new \InvalidArgumentException(
          sprintf('Argument "%s" completion provider configuration must be an array.',
            $argument['machine_name'])
        );
      }
    }
  }

  /**
   * Gets the completion provider plugin manager.
   *
   * @return \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager
   *   The plugin manager.
   */
  private function getCompletionProviderManager(): PromptArgumentCompletionProviderManager {
    return \Drupal::service('plugin.manager.prompt_argument_completion_provider');
  }

  /**
   * Validates message structure and content.
   *
   * @throws \InvalidArgumentException
   *   When message structure or content is invalid.
   */
  private function validateMessages(): void {
    foreach ($this->messages as $index => $message) {
      // PHPDoc guarantees array type, validate role value.
      if (!in_array($message['role'], ['user', 'assistant'], TRUE)) {
        throw new \InvalidArgumentException(sprintf(
          'Message at index %d must have role "user" or "assistant".',
          $index
        ));
      }

      $this->validateMessageContent($message['content'], $index);
    }
  }

  /**
   * Validates message content items.
   *
   * @param array $content
   *   The content array to validate.
   * @param int $message_index
   *   The message index for error reporting.
   *
   * @throws \InvalidArgumentException
   *   When content structure is invalid.
   */
  private function validateMessageContent(array $content, int $message_index): void {
    foreach ($content as $content_index => $item) {
      if (!is_array($item) || empty($item['type'])) {
        throw new \InvalidArgumentException(sprintf(
          'Content item %d in message %d must have a type field.',
          $content_index,
          $message_index
        ));
      }

      match ($item['type']) {
        'text' => $this->validateTextContent($item, $message_index, $content_index),
        'image' => $this->validateImageContent($item, $message_index, $content_index),
        'audio' => $this->validateAudioContent($item, $message_index, $content_index),
        'resource' => $this->validateResourceContent($item, $message_index, $content_index),
        default => throw new \InvalidArgumentException(sprintf(
          'Content item %d in message %d has invalid type "%s".',
          $content_index,
          $message_index,
          $item['type']
        )),
      };
    }
  }

  /**
   * Validates text content structure.
   *
   * @param array $item
   *   The content item.
   * @param int $message_index
   *   The message index.
   * @param int $content_index
   *   The content index.
   *
   * @throws \InvalidArgumentException
   *   When structure is invalid.
   */
  private function validateTextContent(array $item, int $message_index, int $content_index): void {
    if (!isset($item['text']) || !is_string($item['text'])) {
      throw new \InvalidArgumentException(sprintf(
        'Text content item %d in message %d must have a text field.',
        $content_index,
        $message_index
      ));
    }
  }

  /**
   * Validates image content structure.
   *
   * @param array $item
   *   The content item.
   * @param int $message_index
   *   The message index.
   * @param int $content_index
   *   The content index.
   *
   * @throws \InvalidArgumentException
   *   When structure is invalid.
   */
  private function validateImageContent(array $item, int $message_index, int $content_index): void {
    if (!isset($item['data']) || !is_string($item['data'])) {
      throw new \InvalidArgumentException(sprintf(
        'Image content item %d in message %d must have a data field.',
        $content_index,
        $message_index
      ));
    }

    if (!isset($item['mimeType']) || !is_string($item['mimeType'])) {
      throw new \InvalidArgumentException(sprintf(
        'Image content item %d in message %d must have a mimeType field.',
        $content_index,
        $message_index
      ));
    }

    if (!str_starts_with($item['mimeType'], 'image/')) {
      throw new \InvalidArgumentException(sprintf(
        'Image content item %d in message %d has invalid mimeType "%s".',
        $content_index,
        $message_index,
        $item['mimeType']
      ));
    }
  }

  /**
   * Validates audio content structure.
   *
   * @param array $item
   *   The content item.
   * @param int $message_index
   *   The message index.
   * @param int $content_index
   *   The content index.
   *
   * @throws \InvalidArgumentException
   *   When structure is invalid.
   */
  private function validateAudioContent(array $item, int $message_index, int $content_index): void {
    if (!isset($item['data']) || !is_string($item['data'])) {
      throw new \InvalidArgumentException(sprintf(
        'Audio content item %d in message %d must have a data field.',
        $content_index,
        $message_index
      ));
    }

    if (!isset($item['mimeType']) || !is_string($item['mimeType'])) {
      throw new \InvalidArgumentException(sprintf(
        'Audio content item %d in message %d must have a mimeType field.',
        $content_index,
        $message_index
      ));
    }

    if (!str_starts_with($item['mimeType'], 'audio/')) {
      throw new \InvalidArgumentException(sprintf(
        'Audio content item %d in message %d has invalid mimeType "%s".',
        $content_index,
        $message_index,
        $item['mimeType']
      ));
    }
  }

  /**
   * Validates resource content structure.
   *
   * @param array $item
   *   The content item.
   * @param int $message_index
   *   The message index.
   * @param int $content_index
   *   The content index.
   *
   * @throws \InvalidArgumentException
   *   When structure is invalid.
   */
  private function validateResourceContent(array $item, int $message_index, int $content_index): void {
    if (!isset($item['resource']) || !is_array($item['resource'])) {
      throw new \InvalidArgumentException(sprintf(
        'Resource content item %d in message %d must have a resource object.',
        $content_index,
        $message_index
      ));
    }

    $resource = $item['resource'];

    if (!isset($resource['uri']) || !is_string($resource['uri'])) {
      throw new \InvalidArgumentException(sprintf(
        'Resource content item %d in message %d must have a uri field.',
        $content_index,
        $message_index
      ));
    }

    if (isset($resource['mimeType']) && !is_string($resource['mimeType'])) {
      throw new \InvalidArgumentException(sprintf(
        'Resource content item %d in message %d has invalid mimeType.',
        $content_index,
        $message_index
      ));
    }

    if (isset($resource['text']) && !is_string($resource['text'])) {
      throw new \InvalidArgumentException(sprintf(
        'Resource content item %d in message %d has invalid text.',
        $content_index,
        $message_index
      ));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE): void {
    parent::postSave($storage, $update);
    Cache::invalidateTags(['mcp_server:discovery']);
  }

  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities): void {
    parent::postDelete($storage, $entities);
    Cache::invalidateTags(['mcp_server:discovery']);
  }

}
