<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Entity;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface;

/**
 * Defines the MCP Tool Configuration entity.
 *
 * This configuration entity maps Tool API tools to MCP tools, allowing
 * customization of how Drupal tools are exposed to MCP clients.
 *
 * @ConfigEntityType(
 *   id = "mcp_tool_config",
 *   label = @Translation("MCP Tool Configuration"),
 *   label_collection = @Translation("MCP Tool Configurations"),
 *   label_singular = @Translation("MCP tool configuration"),
 *   label_plural = @Translation("MCP tool configurations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count MCP tool configuration",
 *     plural = "@count MCP tool configurations",
 *   ),
 *   handlers = {
 *     "storage" = "Drupal\Core\Config\Entity\ConfigEntityStorage",
 *     "list_builder" = "Drupal\mcp_server\McpToolConfigListBuilder",
 *     "form" = {
 *       "add" = "Drupal\mcp_server\Form\McpToolConfigForm",
 *       "edit" = "Drupal\mcp_server\Form\McpToolConfigForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     },
 *   },
 *   config_prefix = "mcp_tool_config",
 *   admin_permission = "administer mcp tool configurations",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "mcp_tool_name",
 *     "status" = "status",
 *   },
 *   config_export = {
 *     "id",
 *     "mcp_tool_name",
 *     "tool_id",
 *     "description",
 *     "status",
 *     "authentication_mode",
 *     "scopes",
 *   },
 *   links = {
 *     "canonical" = "/admin/config/services/mcp-server/tool/{mcp_tool_config}",
 *     "add-form" = "/admin/config/services/mcp-server/tool/add",
 *     "edit-form" = "/admin/config/services/mcp-server/tool/{mcp_tool_config}/edit",
 *     "delete-form" = "/admin/config/services/mcp-server/tool/{mcp_tool_config}/delete",
 *     "collection" = "/admin/config/services/mcp-server/tools",
 *   }
 * )
 */
final class McpToolConfig extends ConfigEntityBase {

  /**
   * Reference to the Tool API tool ID.
   */
  protected string $tool_id = '';

  /**
   * The MCP tool name.
   */
  protected string $mcp_tool_name = '';

  /**
   * Override for the tool description.
   */
  protected ?string $description = NULL;

  /**
   * The authentication mode for this tool.
   *
   * Possible values: 'required', 'disabled'.
   *
   * @var string
   */
  protected string $authentication_mode = 'disabled';

  /**
   * Array of required OAuth scope names.
   *
   * Scope names like 'content:read', 'content:write', etc. Empty array means
   * no scope restrictions (any authenticated user can execute the tool).
   * Scopes are only validated when authentication_mode is 'required'.
   *
   * @var array
   */
  protected array $scopes = [];

  /**
   * Gets the Tool API tool ID.
   *
   * @return string
   *   The tool ID from Tool API.
   */
  public function getToolId(): string {
    return $this->tool_id;
  }

  /**
   * Gets the tool description override.
   *
   * @return string|null
   *   The description override, or NULL if not set.
   */
  public function getDescription(): ?string {
    return $this->description;
  }

  /**
   * Gets the authentication mode for this tool.
   *
   * @return string
   *   One of: 'required', 'optional', 'disabled'.
   */
  public function getAuthenticationMode(): string {
    return $this->authentication_mode ?? 'disabled';
  }

  /**
   * Sets the authentication mode for this tool.
   *
   * @param string $mode
   *   One of: 'required', 'optional', 'disabled'.
   *
   * @return $this
   *   The entity for chaining.
   *
   * @throws \InvalidArgumentException
   *   If the mode is not one of the allowed values.
   */
  public function setAuthenticationMode(string $mode): self {
    if (!in_array($mode, ['required', 'optional', 'disabled'], TRUE)) {
      throw new \InvalidArgumentException('Invalid authentication mode');
    }
    $this->authentication_mode = $mode;
    return $this;
  }

  /**
   * Checks if authentication is required for this tool.
   *
   * @return bool
   *   TRUE if authentication is required, FALSE otherwise.
   */
  public function isAuthenticationRequired(): bool {
    return $this->getAuthenticationMode() === 'required';
  }

  /**
   * Checks if authentication is enabled for this tool.
   *
   * Returns TRUE for 'required' mode, indicating that authentication metadata
   * should be included in tool responses.
   *
   * @return bool
   *   TRUE if authentication is enabled (required), FALSE if disabled.
   */
  public function requiresAuthentication(): bool {
    return $this->getAuthenticationMode() !== 'disabled';
  }

  /**
   * Gets the required OAuth scopes for this tool.
   *
   * @return array
   *   Array of scope IDs (strings).
   */
  public function getScopes(): array {
    return $this->scopes;
  }

  /**
   * Gets the required OAuth scope names.
   *
   * The scopes field directly stores scope names (e.g., 'content:read'),
   * so this method simply returns them.
   *
   * @return array
   *   Array of scope names (strings).
   */
  public function getScopeNames(): array {
    $scope_entities = $this->entityTypeManager()
      ->getStorage('oauth2_scope')
      ->loadMultiple($this->scopes);
    return array_map(
      static function ($scope_entity): string {
        assert($scope_entity instanceof Oauth2ScopeEntityInterface);
        return $scope_entity->getName();
      },
      $scope_entities,
    );
  }

  /**
   * Sets the required OAuth scopes for this tool.
   *
   * @param array $scopes
   *   Array of scope names (e.g., ['content:read', 'content:write']).
   *
   * @return $this
   *   The entity for chaining.
   */
  public function setScopes(array $scopes): self {
    $this->scopes = $scopes;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    if (empty($this->tool_id)) {
      throw new \InvalidArgumentException('The tool_id field is required.');
    }

    if (empty($this->mcp_tool_name)) {
      throw new \InvalidArgumentException('The mcp_tool_name field is required.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE): void {
    parent::postSave($storage, $update);
    Cache::invalidateTags(['mcp_server:discovery']);
  }

  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities): void {
    parent::postDelete($storage, $entities);
    Cache::invalidateTags(['mcp_server:discovery']);
  }

}
