<?php

declare(strict_types=1);

namespace Drupal\mcp_server\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Event subscriber for MCP authentication error responses.
 *
 * Intercepts JSON-RPC error responses with authentication error codes and
 * sets proper HTTP status (401) and WWW-Authenticate headers per OAuth 2.0.
 */
final class AuthenticationErrorSubscriber implements EventSubscriberInterface {

  /**
   * JSON-RPC error code for authentication failures.
   *
   * Reserved error codes:
   * - -32001: Authentication required (PRD 1)
   * - -32002: Authorization failed / insufficient scopes (PRD 2)
   */
  private const ERROR_CODE_AUTHENTICATION_REQUIRED = -32001;

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run after routing but before response finalization.
    return [
      KernelEvents::RESPONSE => ['onResponse', 0],
    ];
  }

  /**
   * Handles response events to modify authentication errors.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $response = $event->getResponse();

    // Only process JSON responses from MCP endpoints.
    if (!$response instanceof JsonResponse) {
      return;
    }

    $content = $response->getContent();
    if ($content === FALSE) {
      return;
    }

    $data = json_decode($content, TRUE);
    if (!is_array($data)) {
      return;
    }

    // Check if this is a JSON-RPC error response with authentication error.
    if (!$this->isAuthenticationError($data)) {
      return;
    }

    // Set HTTP 401 status and WWW-Authenticate header.
    $response->setStatusCode(401);
    $response->headers->set('WWW-Authenticate', 'Bearer realm="mcp_server"');
  }

  /**
   * Checks if the response data contains an authentication error.
   *
   * @param array $data
   *   The decoded JSON response data.
   *
   * @return bool
   *   TRUE if this is an authentication error, FALSE otherwise.
   */
  private function isAuthenticationError(array $data): bool {
    // JSON-RPC error format: {"jsonrpc": "2.0", "error": {...}, "id": ...}.
    if (!isset($data['jsonrpc']) || $data['jsonrpc'] !== '2.0') {
      return FALSE;
    }

    if (!isset($data['error']) || !is_array($data['error'])) {
      return FALSE;
    }

    if (!isset($data['error']['code'])) {
      return FALSE;
    }

    return $data['error']['code'] === self::ERROR_CODE_AUTHENTICATION_REQUIRED;
  }

}
