<?php

declare(strict_types=1);

namespace Drupal\mcp_server\EventSubscriber;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponseInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Adds MCP cache tags to OAuth metadata endpoint responses.
 *
 * This subscriber ensures that the OAuth Protected Resource Metadata endpoint
 * (/.well-known/oauth-protected-resource) response is invalidated whenever
 * MCP tool configurations are modified. This is critical for both tests and
 * production environments where tools can be added, removed, or modified
 * at runtime.
 */
final class MetadataCacheSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run early in the response phase, before cache is finalized.
    // Higher priority = runs earlier. Default caching runs at priority 0.
    return [
      KernelEvents::RESPONSE => ['onResponse', 100],
    ];
  }

  /**
   * Adds MCP cache tags to metadata endpoint responses.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    // Only process main requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();
    $response = $event->getResponse();

    // Check if this is the OAuth metadata endpoint.
    $path = $request->getPathInfo();
    if (!str_contains($path, 'oauth-protected-resource')) {
      return;
    }

    // Only add cache tags to cacheable responses.
    if (!$response instanceof CacheableResponseInterface) {
      return;
    }

    // Add the MCP discovery cache tag.
    // This tag is invalidated when MCP tool configs are saved/deleted.
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->addCacheTags(['mcp_server:discovery']);
    $response->addCacheableDependency($cache_metadata);
  }

}
