<?php

declare(strict_types=1);

namespace Drupal\mcp_server\EventSubscriber;

use Drupal\mcp_server\Service\OAuthScopeDiscoveryService;
use Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvent;
use Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Extends RFC 9728 Protected Resource Metadata with MCP tool scopes.
 *
 * This subscriber enriches the OAuth 2.0 Protected Resource Metadata endpoint
 * (/.well-known/oauth-protected-resource) with scopes discovered from enabled
 * MCP tool configurations. It merges MCP scopes with any existing
 * scopes_supported array and ensures deduplication.
 */
final class ResourceMetadataSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a ResourceMetadataSubscriber.
   *
   * @param \Drupal\mcp_server\Service\OAuthScopeDiscoveryService $scopeDiscovery
   *   The OAuth scope discovery service.
   */
  public function __construct(
    private readonly OAuthScopeDiscoveryService $scopeDiscovery,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ResourceMetadataEvents::BUILD => 'onBuild',
    ];
  }

  /**
   * Responds to resource metadata build events.
   *
   * Adds scopes_supported field to the metadata response with MCP tool scopes.
   * Merges with existing scopes if present and deduplicates.
   *
   * @param \Drupal\simple_oauth_server_metadata\Event\ResourceMetadataEvent $event
   *   The resource metadata event.
   */
  public function onBuild(ResourceMetadataEvent $event): void {
    // Get MCP scopes from discovery service.
    $mcp_scopes = $this->scopeDiscovery->getScopesSupported();

    // Early return if no MCP scopes are configured.
    if (empty($mcp_scopes)) {
      return;
    }

    // Get existing scopes_supported if present.
    $existing_scopes = $event->metadata['scopes_supported'] ?? [];

    // Merge and deduplicate scopes.
    $all_scopes = array_merge($existing_scopes, $mcp_scopes);
    $unique_scopes = array_values(array_unique($all_scopes));

    // Set the merged scopes back to metadata.
    $event->addMetadataField('scopes_supported', $unique_scopes);
  }

}
