<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Exception;

/**
 * Exception thrown when authentication is required but not provided.
 */
final class AuthenticationRequiredException extends \RuntimeException {

  /**
   * JSON-RPC error code for authentication required.
   */
  public const ERROR_CODE = -32001;

  /**
   * Constructs an AuthenticationRequiredException.
   *
   * @param string $toolName
   *   The name of the tool requiring authentication.
   * @param string $authenticationMode
   *   The authentication mode (e.g., 'required').
   * @param \Throwable|null $previous
   *   The previous throwable for chaining.
   */
  public function __construct(
    private readonly string $toolName,
    private readonly string $authenticationMode = 'required',
    ?\Throwable $previous = NULL,
  ) {
    $message = sprintf(
      'Authentication required for tool "%s" (mode: %s)',
      $toolName,
      $authenticationMode,
    );

    parent::__construct($message, self::ERROR_CODE, $previous);
  }

  /**
   * Gets the tool name.
   */
  public function getToolName(): string {
    return $this->toolName;
  }

  /**
   * Gets the authentication mode.
   */
  public function getAuthenticationMode(): string {
    return $this->authenticationMode;
  }

}
