<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\mcp_server\Entity\McpPromptConfig;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for MCP Prompt Configuration entities.
 */
final class McpPromptConfigForm extends EntityForm {

  use PluginSelectionFormTrait;

  /**
   * The completion provider plugin manager.
   */
  protected PromptArgumentCompletionProviderManager $completionProviderManager;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs an McpPromptConfigForm.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    PromptArgumentCompletionProviderManager $completionProviderManager,
    FileSystemInterface $fileSystem,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->completionProviderManager = $completionProviderManager;
    $this->fileSystem = $fileSystem;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.prompt_argument_completion_provider'),
      $container->get('file_system'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    $entity = $this->entity;
    if (!$entity instanceof McpPromptConfig) {
      return $form;
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prompt Name'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('The name of this prompt as exposed to MCP clients.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
        'source' => ['label'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#maxlength' => 255,
      '#default_value' => $entity->getTitle(),
      '#description' => $this->t('Optional title for the prompt.'),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->getDescription(),
      '#description' => $this->t('Optional description of what this prompt does.'),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
      '#description' => $this->t('When enabled, this prompt will be exposed to MCP clients.'),
    ];

    // Arguments fieldset with AJAX.
    $form['arguments'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Arguments'),
      '#description' => $this->t('Define arguments that can be provided to this prompt.'),
      '#prefix' => '<div id="arguments-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    // Get arguments from form state (if set during AJAX) or entity.
    // Form state is used to preserve state during AJAX operations.
    $arguments = $form_state->get('arguments') ?? $entity->getArguments();
    if (!$form_state->has('arguments')) {
      $form_state->set('arguments', $arguments);
    }

    foreach ($arguments as $index => $argument) {
      $form['arguments'][$index] = $this->buildArgumentElement($argument, $index, $form_state);
    }

    $form['arguments']['add_argument'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add argument'),
      '#submit' => ['::addArgumentSubmit'],
      '#ajax' => [
        'callback' => '::argumentsAjaxCallback',
        'wrapper' => 'arguments-wrapper',
      ],
    ];

    // Messages fieldset with AJAX.
    $form['messages'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Messages'),
      '#description' => $this->t('Define the message sequence for this prompt.'),
      '#prefix' => '<div id="messages-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    $messages = $form_state->get('messages') ?? $entity->getMessages();
    $form_state->set('messages', $messages);

    foreach ($messages as $index => $message) {
      $form['messages'][$index] = $this->buildMessageElement($message, $index, $form_state);
    }

    $form['messages']['add_message'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add message'),
      '#submit' => ['::addMessageSubmit'],
      '#ajax' => [
        'callback' => '::messagesAjaxCallback',
        'wrapper' => 'messages-wrapper',
      ],
    ];

    return $form;
  }

  /**
   * Builds a single argument form element.
   *
   * @param array $argument
   *   The argument data.
   * @param int $index
   *   The argument index.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Form element array.
   */
  private function buildArgumentElement(array $argument, int $index, FormStateInterface $form_state): array {
    $element = [
      '#type' => 'fieldset',
      '#title' => $this->t('Argument @num', ['@num' => $index + 1]),
    ];

    $element['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $argument['label'] ?? '',
      '#required' => TRUE,
      '#weight' => -10,
    ];

    $element['machine_name'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Machine name'),
      '#default_value' => $argument['machine_name'] ?? '',
      '#machine_name' => [
        'source' => ['arguments', $index, 'label'],
        'exists' => [$this, 'argumentMachineNameExists'],
        'standalone' => TRUE,
      ],
      '#required' => TRUE,
      '#weight' => -9,
    ];

    $element['description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Description'),
      '#default_value' => $argument['description'] ?? '',
      '#required' => TRUE,
      '#weight' => -8,
    ];

    $element['required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Required'),
      '#default_value' => $argument['required'] ?? FALSE,
      '#weight' => -7,
    ];

    // Inject plugin selector for completion providers.
    $this->injectPluginSelector(
      $element,
      $form_state,
      $argument['completion_providers'] ?? [],
      'completion_providers',
      $index,
      TRUE
    );

    $element['remove'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove'),
      '#name' => 'remove_argument_' . $index,
      '#submit' => ['::removeArgumentSubmit'],
      '#ajax' => [
        'callback' => '::argumentsAjaxCallback',
        'wrapper' => 'arguments-wrapper',
      ],
      '#argument_index' => $index,
    ];

    return $element;
  }

  /**
   * Builds a single message form element.
   *
   * @param array $message
   *   The message data.
   * @param int $index
   *   The message index.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Form element array.
   */
  private function buildMessageElement(array $message, int $index, FormStateInterface $form_state): array {
    $wrapper_id = 'message-' . $index . '-wrapper';

    // Fieldset with wrapper div for AJAX replacement.
    $element = [
      '#type' => 'fieldset',
      '#title' => $this->t('Message @num', ['@num' => $index + 1]),
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    $element['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Role'),
      '#options' => [
        'user' => $this->t('User'),
        'assistant' => $this->t('Assistant'),
      ],
      '#default_value' => $message['role'] ?? 'user',
      '#required' => TRUE,
    ];

    // Content items.
    $element['content'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content'),
      '#tree' => TRUE,
    ];

    $content = $message['content'] ?? [];
    foreach ($content as $content_index => $content_item) {
      $element['content'][$content_index] = $this->buildContentElement(
        $content_item,
        $index,
        $content_index,
        $form_state
      );
    }

    $element['content']['add_content'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add content'),
      '#name' => 'add_content_' . $index,
      '#submit' => ['::addContentSubmit'],
      '#ajax' => [
        'callback' => '::messageAjaxCallback',
        'wrapper' => 'message-' . $index . '-wrapper',
      ],
      '#message_index' => $index,
    ];

    $element['remove'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove message'),
      '#name' => 'remove_message_' . $index,
      '#submit' => ['::removeMessageSubmit'],
      '#ajax' => [
        'callback' => '::messagesAjaxCallback',
        'wrapper' => 'messages-wrapper',
      ],
      '#message_index' => $index,
    ];

    return $element;
  }

  /**
   * Builds a single content item form element.
   *
   * @param array $content_item
   *   The content item data.
   * @param int $message_index
   *   The message index.
   * @param int $content_index
   *   The content index.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   Form element array.
   */
  private function buildContentElement(array $content_item, int $message_index, int $content_index, FormStateInterface $form_state): array {
    // Check user input first (for AJAX rebuilds), then submitted values, then
    // default.
    $user_input = $form_state->getUserInput();
    $content_type = $user_input['messages'][$message_index]['content'][$content_index]['type']
      ?? $form_state->getValue(['messages', $message_index, 'content', $content_index, 'type'])
      ?? $content_item['type']
      ?? 'text';

    $element = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content item @num', ['@num' => $content_index + 1]),
    ];

    $element['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Type'),
      '#options' => [
        'text' => $this->t('Text'),
        'image' => $this->t('Image'),
        'audio' => $this->t('Audio'),
        'resource' => $this->t('Resource'),
      ],
      '#default_value' => $content_type,
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::messageAjaxCallback',
        'wrapper' => 'message-' . $message_index . '-wrapper',
      ],
      '#message_index' => $message_index,
    ];

    // Dynamic content fields based on type.
    match ($content_type) {
      'text' => $element += $this->buildTextContentFields($content_item),
      'image' => $element += $this->buildImageContentFields($content_item, $message_index, $content_index, $form_state),
      'audio' => $element += $this->buildAudioContentFields($content_item, $message_index, $content_index, $form_state),
      'resource' => $element += $this->buildResourceContentFields($content_item),
      default => NULL,
    };

    $element['remove_content'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove'),
      '#name' => 'remove_content_' . $message_index . '_' . $content_index,
      '#submit' => ['::removeContentSubmit'],
      '#ajax' => [
        'callback' => '::messageAjaxCallback',
        'wrapper' => 'message-' . $message_index . '-wrapper',
      ],
      '#message_index' => $message_index,
      '#content_index' => $content_index,
    ];

    return $element;
  }

  /**
   * Builds text content fields.
   */
  private function buildTextContentFields(array $content_item): array {
    return [
      'text' => [
        '#type' => 'textarea',
        '#title' => $this->t('Text'),
        '#default_value' => $content_item['text'] ?? '',
        '#required' => TRUE,
      ],
    ];
  }

  /**
   * Builds image content fields.
   */
  private function buildImageContentFields(array $content_item, int $message_index, int $content_index, FormStateInterface $form_state): array {
    $user_input = $form_state->getUserInput();

    // Default to 'manual' if mimeType and data are already populated.
    $has_data = !empty($content_item['mimeType']) && !empty($content_item['data']);
    $default_method = $has_data ? 'manual' : 'upload';

    $input_method_key = $user_input['messages'][$message_index]['content'][$content_index]['input_method']
      ?? $content_item['input_method']
      ?? $default_method;

    $fields = [
      'input_method' => [
        '#type' => 'radios',
        '#title' => $this->t('Input Method'),
        '#options' => [
          'upload' => $this->t('Upload file'),
          'manual' => $this->t('Manual entry (MIME type + base64 data)'),
        ],
        '#default_value' => $input_method_key,
        '#required' => TRUE,
        '#ajax' => [
          'callback' => '::messageAjaxCallback',
          'wrapper' => 'message-' . $message_index . '-wrapper',
        ],
      ],
    ];

    if ($input_method_key === 'upload') {
      $fields['file_upload'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Image File'),
        '#upload_location' => 'temporary://mcp_prompts',
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'png jpg jpeg gif',
          ],
        ],
        '#description' => $this->t('Upload an image file. Will be converted to base64.'),
      ];
    }
    else {
      $fields['mimeType'] = [
        '#type' => 'textfield',
        '#title' => $this->t('MIME Type'),
        '#default_value' => $content_item['mimeType'] ?? '',
        '#required' => TRUE,
        '#description' => $this->t('E.g., image/png, image/jpeg'),
      ];
      $fields['data'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Base64 Data'),
        '#default_value' => $content_item['data'] ?? '',
        '#required' => TRUE,
        '#description' => $this->t('Base64 encoded image data.'),
      ];
    }

    return $fields;
  }

  /**
   * Builds audio content fields.
   */
  private function buildAudioContentFields(array $content_item, int $message_index, int $content_index, FormStateInterface $form_state): array {
    $user_input = $form_state->getUserInput();

    // Default to 'manual' if mimeType and data are already populated.
    $has_data = !empty($content_item['mimeType']) && !empty($content_item['data']);
    $default_method = $has_data ? 'manual' : 'upload';

    $input_method_key = $user_input['messages'][$message_index]['content'][$content_index]['input_method']
      ?? $content_item['input_method']
      ?? $default_method;

    $fields = [
      'input_method' => [
        '#type' => 'radios',
        '#title' => $this->t('Input Method'),
        '#options' => [
          'upload' => $this->t('Upload file'),
          'manual' => $this->t('Manual entry (MIME type + base64 data)'),
        ],
        '#default_value' => $input_method_key,
        '#required' => TRUE,
        '#ajax' => [
          'callback' => '::messageAjaxCallback',
          'wrapper' => 'message-' . $message_index . '-wrapper',
        ],
      ],
    ];

    if ($input_method_key === 'upload') {
      $fields['file_upload'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Audio File'),
        '#upload_location' => 'temporary://mcp_prompts',
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'mp3 wav ogg',
          ],
        ],
        '#description' => $this->t('Upload an audio file. Will be converted to base64.'),
      ];
    }
    else {
      $fields['mimeType'] = [
        '#type' => 'textfield',
        '#title' => $this->t('MIME Type'),
        '#default_value' => $content_item['mimeType'] ?? '',
        '#required' => TRUE,
        '#description' => $this->t('E.g., audio/wav, audio/mp3'),
      ];
      $fields['data'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Base64 Data'),
        '#default_value' => $content_item['data'] ?? '',
        '#required' => TRUE,
        '#description' => $this->t('Base64 encoded audio data.'),
      ];
    }

    return $fields;
  }

  /**
   * Builds resource content fields.
   */
  private function buildResourceContentFields(array $content_item): array {
    $resource = $content_item['resource'] ?? [];

    return [
      'resource' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Resource'),
        '#tree' => TRUE,
        'uri' => [
          '#type' => 'textfield',
          '#title' => $this->t('URI'),
          '#default_value' => $resource['uri'] ?? '',
          '#required' => TRUE,
        ],
        'mimeType' => [
          '#type' => 'textfield',
          '#title' => $this->t('MIME Type'),
          '#default_value' => $resource['mimeType'] ?? '',
        ],
        'text' => [
          '#type' => 'textarea',
          '#title' => $this->t('Text'),
          '#default_value' => $resource['text'] ?? '',
        ],
      ],
    ];
  }

  /**
   * AJAX callback for arguments fieldset.
   */
  public function argumentsAjaxCallback(array &$form, FormStateInterface $form_state): array {
    return $form['arguments'];
  }

  /**
   * AJAX callback for messages fieldset.
   */
  public function messagesAjaxCallback(array &$form, FormStateInterface $form_state): array {
    return $form['messages'];
  }

  /**
   * AJAX callback for a single message.
   */
  public function messageAjaxCallback(array &$form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $message_index = $triggering_element['#message_index'] ?? NULL;

    if ($message_index !== NULL && isset($form['messages'][$message_index])) {
      return $form['messages'][$message_index];
    }

    return $form['messages'];
  }

  /**
   * Submit handler for adding an argument.
   */
  public function addArgumentSubmit(array &$form, FormStateInterface $form_state): void {
    $arguments = $form_state->get('arguments') ?? [];
    $arguments[] = [
      'label' => '',
      'machine_name' => '',
      'description' => '',
      'required' => FALSE,
    ];
    $form_state->set('arguments', $arguments);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing an argument.
   */
  public function removeArgumentSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $index = $triggering_element['#argument_index'] ?? NULL;

    if ($index !== NULL) {
      $arguments = $form_state->get('arguments') ?? [];
      unset($arguments[$index]);
      $form_state->set('arguments', array_values($arguments));
      $form_state->setRebuild();
    }
  }

  /**
   * Submit handler for adding a message.
   */
  public function addMessageSubmit(array &$form, FormStateInterface $form_state): void {
    $messages = $form_state->get('messages') ?? [];
    $messages[] = [
      'role' => 'user',
      'content' => [],
    ];
    $form_state->set('messages', $messages);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing a message.
   */
  public function removeMessageSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $index = $triggering_element['#message_index'] ?? NULL;

    if ($index !== NULL) {
      $messages = $form_state->get('messages') ?? [];
      unset($messages[$index]);
      $form_state->set('messages', array_values($messages));
      $form_state->setRebuild();
    }
  }

  /**
   * Submit handler for adding content to a message.
   */
  public function addContentSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $message_index = $triggering_element['#message_index'] ?? NULL;

    if ($message_index !== NULL) {
      $messages = $form_state->get('messages') ?? [];
      if (!isset($messages[$message_index]['content'])) {
        $messages[$message_index]['content'] = [];
      }
      $messages[$message_index]['content'][] = [
        'type' => 'text',
        'text' => '',
      ];
      $form_state->set('messages', $messages);
      $form_state->setRebuild();
    }
  }

  /**
   * Submit handler for removing content from a message.
   */
  public function removeContentSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $message_index = $triggering_element['#message_index'] ?? NULL;
    $content_index = $triggering_element['#content_index'] ?? NULL;

    if ($message_index !== NULL && $content_index !== NULL) {
      $messages = $form_state->get('messages') ?? [];
      if (isset($messages[$message_index]['content'][$content_index])) {
        unset($messages[$message_index]['content'][$content_index]);
        $messages[$message_index]['content'] = array_values($messages[$message_index]['content']);
        $form_state->set('messages', $messages);
        $form_state->setRebuild();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate arguments.
    $arguments = $form_state->getValue('arguments') ?? [];
    foreach ($arguments as $index => $argument) {
      if (is_array($argument) && !isset($argument['remove'])) {
        if (empty($argument['label'])) {
          $form_state->setErrorByName("arguments][$index][label", $this->t('Argument label is required.'));
        }
        if (empty($argument['machine_name'])) {
          $form_state->setErrorByName("arguments][$index][machine_name", $this->t('Argument machine name is required.'));
        }
        if (empty($argument['description'])) {
          $form_state->setErrorByName("arguments][$index][description", $this->t('Argument description is required.'));
        }

        // Validate completion providers using trait.
        $this->validatePluginForm($form, $form_state, 'completion_providers', $index, TRUE);
      }
    }

    // Validate messages.
    $messages = $form_state->getValue('messages') ?? [];
    foreach ($messages as $msg_index => $message) {
      if (!is_array($message) || isset($message['remove'])) {
        continue;
      }

      $content = $message['content'] ?? [];
      foreach ($content as $content_index => $content_item) {
        if (!is_array($content_item) || isset($content_item['remove_content'])) {
          continue;
        }

        $type = $content_item['type'] ?? '';
        match ($type) {
          'text' => $this->validateTextContent($content_item, $msg_index, $content_index, $form_state),
          'image' => $this->validateImageContent($content_item, $msg_index, $content_index, $form_state),
          'audio' => $this->validateAudioContent($content_item, $msg_index, $content_index, $form_state),
          'resource' => $this->validateResourceContent($content_item, $msg_index, $content_index, $form_state),
          default => NULL,
        };
      }
    }
  }

  /**
   * Validates text content.
   */
  private function validateTextContent(array $content_item, int $msg_index, int $content_index, FormStateInterface $form_state): void {
    if (empty($content_item['text'])) {
      $form_state->setErrorByName("messages][$msg_index][content][$content_index][text", $this->t('Text content is required.'));
    }
  }

  /**
   * Validates image content.
   */
  private function validateImageContent(array $content_item, int $msg_index, int $content_index, FormStateInterface $form_state): void {
    $input_method = $content_item['input_method'] ?? 'upload';

    if ($input_method === 'upload') {
      if (empty($content_item['file_upload'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][file_upload", $this->t('Image file upload is required.'));
      }
    }
    else {
      if (empty($content_item['mimeType'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][mimeType", $this->t('MIME type is required.'));
      }
      if (empty($content_item['data'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][data", $this->t('Base64 data is required.'));
      }
    }
  }

  /**
   * Validates audio content.
   */
  private function validateAudioContent(array $content_item, int $msg_index, int $content_index, FormStateInterface $form_state): void {
    $input_method = $content_item['input_method'] ?? 'upload';

    if ($input_method === 'upload') {
      if (empty($content_item['file_upload'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][file_upload", $this->t('Audio file upload is required.'));
      }
    }
    else {
      if (empty($content_item['mimeType'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][mimeType", $this->t('MIME type is required.'));
      }
      if (empty($content_item['data'])) {
        $form_state->setErrorByName("messages][$msg_index][content][$content_index][data", $this->t('Base64 data is required.'));
      }
    }
  }

  /**
   * Validates resource content.
   */
  private function validateResourceContent(array $content_item, int $msg_index, int $content_index, FormStateInterface $form_state): void {
    $resource = $content_item['resource'] ?? [];
    if (empty($resource['uri'])) {
      $form_state->setErrorByName("messages][$msg_index][content][$content_index][resource][uri", $this->t('Resource URI is required.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    // Process arguments - remove add/remove buttons.
    $arguments = array_filter(
      $form_state->getValue('arguments') ?? [],
      fn($arg) => is_array($arg) && !isset($arg['remove'])
    );

    // Process completion providers first, then clean up arguments.
    $processed_arguments = [];
    foreach ($arguments as $index => $arg) {
      unset($arg['remove']);
      $processed_arg = [
        'label' => $arg['label'],
        'machine_name' => $arg['machine_name'],
        'description' => $arg['description'],
        'required' => (bool) ($arg['required'] ?? FALSE),
      ];

      // Process completion providers using trait.
      $providers = $this->updatePluginConfiguration($form, $form_state, 'completion_providers', $index, TRUE);
      if (!empty($providers)) {
        $processed_arg['completion_providers'] = $providers;
      }

      $processed_arguments[] = $processed_arg;
    }
    $arguments = $processed_arguments;

    // Process messages.
    $messages = array_filter(
      $form_state->getValue('messages') ?? [],
      fn($msg) => is_array($msg) && !isset($msg['remove'])
    );

    $messages = array_map(function ($message) {
      $content = array_filter(
        $message['content'] ?? [],
        fn($item) => is_array($item) && !isset($item['remove_content']) && !isset($item['add_content'])
      );

      $content = array_map(function ($item) {
        return $this->processContentItem($item);
      }, $content);

      unset($message['remove'], $message['content']);

      return [
        'role' => $message['role'],
        'content' => array_values($content),
      ];
    }, $messages);

    $entity = $this->entity;
    assert($entity instanceof McpPromptConfig);
    $entity->set('arguments', $arguments);
    $entity->set('messages', array_values($messages));
  }

  /**
   * Processes a content item to match the expected structure.
   *
   * Handles file uploads by converting to base64 and cleaning up uploaded
   * files.
   *
   * @param array $item
   *   The content item from form values.
   *
   * @return array
   *   The processed content item ready for entity storage.
   */
  private function processContentItem(array $item): array {
    $type = $item['type'] ?? 'text';

    return match ($type) {
      'text' => [
        'type' => 'text',
        'text' => $item['text'] ?? '',
      ],
      'image' => $this->processMediaContent($item, 'image/'),
      'audio' => $this->processMediaContent($item, 'audio/'),
      'resource' => [
        'type' => 'resource',
        'resource' => [
          'uri' => $item['resource']['uri'] ?? '',
          'mimeType' => $item['resource']['mimeType'] ?? NULL,
          'text' => $item['resource']['text'] ?? NULL,
        ],
      ],
      default => ['type' => $type],
    };
  }

  /**
   * Processes image or audio content with file upload support.
   *
   * @param array $item
   *   The content item.
   * @param string $mime_prefix
   *   The expected MIME type prefix.
   *
   * @return array
   *   Processed media content.
   */
  private function processMediaContent(array $item, string $mime_prefix): array {
    $data = $item['data'] ?? '';
    $mimeType = $item['mimeType'] ?? '';

    // Process file upload if present.
    if (!empty($item['file_upload'][0])) {
      $file_storage = $this->entityTypeManager->getStorage('file');
      $file = $file_storage->load($item['file_upload'][0]);
      if ($file) {
        $file_uri = $file->getFileUri();
        // Validate file exists and is readable using Drupal's file system.
        $real_path = $this->fileSystem->realpath($file_uri);
        if ($real_path && is_readable($real_path)) {
          // Read file contents using stream wrapper URI.
          $file_contents = file_get_contents($file_uri);
          if ($file_contents !== FALSE) {
            $data = base64_encode($file_contents);
            if (!$mimeType) {
              $mimeType = $file->getMimeType();
            }
          }
        }
        // Delete the temporary file.
        $file->delete();
      }
    }

    $type = str_starts_with($mime_prefix, 'image/') ? 'image' : 'audio';

    $result = [
      'type' => $type,
      'data' => $data,
    ];

    // Only include mimeType if it's not empty to allow entity validation to
    // catch the error.
    if (!empty($mimeType)) {
      $result['mimeType'] = $mimeType;
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $entity = $this->entity;

    $message_args = ['%label' => $entity->label()];
    $message = $result === SAVED_NEW
      ? $this->t('Created new MCP prompt configuration %label.', $message_args)
      : $this->t('Updated MCP prompt configuration %label.', $message_args);
    $this->messenger()->addStatus($message);

    $form_state->setRedirectUrl($entity->toUrl('collection'));
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  protected function completionProviderManager(): PromptArgumentCompletionProviderManager {
    return $this->completionProviderManager;
  }

  /**
   * Checks if an entity with the given ID exists.
   */
  public function exist(string $id): bool {
    $entity = $this->entityTypeManager
      ->getStorage('mcp_prompt_config')
      ->getQuery()
      ->condition('id', $id)
      ->accessCheck(FALSE)
      ->execute();
    return (bool) $entity;
  }

  /**
   * Checks if an argument machine name already exists in the current prompt.
   *
   * @param string $machine_name
   *   The machine name to check.
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE if the machine name exists, FALSE otherwise.
   */
  public function argumentMachineNameExists(string $machine_name, array $element, FormStateInterface $form_state): bool {
    $arguments = $form_state->getValue('arguments') ?? [];
    $current_index = $element['#array_parents'][1] ?? NULL;

    foreach ($arguments as $index => $argument) {
      // Skip if not an array (can happen during validation).
      if (!is_array($argument)) {
        continue;
      }
      // Skip the current argument being edited.
      if ($index === $current_index) {
        continue;
      }
      // Cast to string to handle both string and TranslatableMarkup.
      if ((string) ($argument['machine_name'] ?? '') === $machine_name) {
        return TRUE;
      }
    }
    return FALSE;
  }

}
