<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\mcp_server\ToolApiDiscovery;
use Drupal\simple_oauth\Entity\Oauth2ScopeEntityInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for MCP Tool Configuration entities.
 */
final class McpToolConfigForm extends EntityForm {

  /**
   * Constructs an McpToolConfigForm.
   */
  public function __construct(
    private readonly ToolApiDiscovery $toolApiDiscovery,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('mcp_server.tool_api_discovery'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    $entity = $this->entity;
    if (!$entity instanceof McpToolConfig) {
      return $form;
    }

    $form['mcp_tool_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('MCP Tool Name'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('The name of this tool as exposed to MCP clients. Also serves as the administrative label.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
        'source' => ['mcp_tool_name'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    // Tool ID autocomplete.
    $form['tool_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tool API Tool ID'),
      '#default_value' => $entity->get('tool_id'),
      '#required' => TRUE,
      '#description' => $this->t('The ID of the Tool API tool to expose (e.g., module_name:tool_name)'),
      '#autocomplete_route_name' => 'mcp_server.tool_autocomplete',
    ];

    // Description override.
    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#description' => $this->t('Override the tool description for AI context.'),
    ];

    // Authentication mode.
    $form['authentication_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Mode'),
      '#description' => $this->t('Controls whether this tool requires authentication. <strong>Required</strong>: Rejects anonymous requests. <strong>Optional</strong>: Accepts both authenticated and anonymous requests (default). <strong>Disabled</strong>: Skips authentication checks entirely.'),
      '#options' => [
        'required' => $this->t('Required - Authentication mandatory'),
        'optional' => $this->t('Optional - Authentication allowed but not required (default)'),
        'disabled' => $this->t('Disabled - No authentication checks'),
      ],
      '#default_value' => $entity->getAuthenticationMode(),
      '#required' => TRUE,
    ];

    // OAuth scopes.
    $form['scopes'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Required OAuth Scopes'),
      '#description' => $this->t('Select OAuth scopes required to execute this tool. Leave empty to allow any authenticated user. Only applicable when authentication mode is "required" or "optional".'),
      '#options' => $this->getAvailableScopes(),
      '#default_value' => $entity->getScopes(),
      '#states' => [
        'visible' => [
          ':input[name="authentication_mode"]' => [
            ['value' => 'required'],
            ['value' => 'optional'],
          ],
        ],
      ],
    ];

    // Status.
    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
      '#description' => $this->t('When enabled, this tool will be exposed to MCP clients.'),
    ];

    // Tool schema preview (if editing existing).
    if (!$entity->isNew() && $entity->get('tool_id')) {
      $toolDef = $this->toolApiDiscovery->getToolDefinition($entity->get('tool_id'));
      if ($toolDef) {
        $form['schema_preview'] = [
          '#type' => 'details',
          '#title' => $this->t('Tool Schema'),
          '#open' => FALSE,
          'content' => [
            '#markup' => '<pre>' . htmlspecialchars(print_r($toolDef, TRUE)) . '</pre>',
          ],
        ];
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $toolId = $form_state->getValue('tool_id');
    if ($toolId) {
      $toolDef = $this->toolApiDiscovery->getToolDefinition($toolId);
      if (!$toolDef) {
        $form_state->setErrorByName('tool_id', $this->t('Tool ID "@tool" not found in Tool API.', [
          '@tool' => $toolId,
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    // Filter out unchecked values (checkboxes return 0 for unchecked).
    $scopes = array_filter($form_state->getValue('scopes'));
    $entity = $this->entity;
    assert($entity instanceof McpToolConfig);
    $entity->setScopes(array_keys($scopes));
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $entity = $this->entity;

    $message_args = ['%label' => $entity->label()];
    $message = $result === SAVED_NEW
      ? $this->t('Created new MCP tool configuration %label.', $message_args)
      : $this->t('Updated MCP tool configuration %label.', $message_args);
    $this->messenger()->addStatus($message);

    $form_state->setRedirectUrl($entity->toUrl('collection'));
    return $result;
  }

  /**
   * Gets available OAuth scopes as form options.
   *
   * Loads all oauth2_scope entities and formats them as options for the
   * checkboxes field. Each option displays the scope label and description.
   *
   * @return array
   *   An associative array where keys are scope IDs and values are formatted
   *   strings showing label and description. Empty array if scopes cannot be
   *   loaded.
   */
  protected function getAvailableScopes(): array {
    try {
      $scope_storage = $this->entityTypeManager->getStorage('oauth2_scope');
      $scopes = $scope_storage->loadMultiple();

      $options = [];
      foreach ($scopes as $scope) {
        assert($scope instanceof Oauth2ScopeEntityInterface);
        $options[$scope->id()] = sprintf(
          '%s - %s',
          $scope->label(),
          $scope->get('description')
        );
      }

      return $options;
    }
    catch (\Exception $e) {
      $this->loggerFactory
        ->get('mcp_server')
        ->error('Failed to load OAuth scopes: @message', [
          '@message' => $e->getMessage(),
        ]);
      return [];
    }
  }

  /**
   * Checks if an entity with the given ID exists.
   */
  public function exist(string $id): bool {
    $entity = $this->entityTypeManager
      ->getStorage('mcp_tool_config')
      ->getQuery()
      ->condition('id', $id)
      ->accessCheck(FALSE)
      ->execute();
    return (bool) $entity;
  }

}
