<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;

/**
 * Provides reusable AJAX-based plugin selection with configuration forms.
 *
 * This trait allows forms to inject plugin selectors that dynamically display
 * configuration forms for selected plugins using AJAX. It handles the complete
 * lifecycle of plugin selection, validation, and submission.
 *
 * Requirements:
 * - Implementing class must provide completionProviderManager() method
 *   that returns a PromptArgumentCompletionProviderManager instance.
 * - Implementing class should use StringTranslationTrait for $this->t().
 *
 * @see \Drupal\ab_tests\Form\PluginSelectionFormTrait
 */
trait PluginSelectionFormTrait {

  use StringTranslationTrait;

  /**
   * AJAX callback for plugin selector changes.
   *
   * Returns the configuration wrapper element that contains the plugin
   * configuration forms for all selected plugins.
   *
   * @param array $form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The configuration wrapper element to be rendered.
   */
  public function pluginSelectionAjaxCallback(array &$form, FormStateInterface $form_state): array {
    $trigger = $form_state->getTriggeringElement();
    // Checkboxes create child elements, so we need to go up 2 levels.
    // For example: ['arguments', 0, 'completion_providers', 'static_list']
    // becomes: ['arguments', 0, 'config_wrapper'].
    $parents = [
      ...array_slice($trigger['#array_parents'], 0, -2),
      'config_wrapper',
    ];
    $element = NestedArray::getValue($form, $parents);

    // Return empty container if element not found (defensive coding).
    if (!is_array($element)) {
      return [
        '#type' => 'container',
        '#attributes' => ['id' => $trigger['#ajax']['wrapper'] ?? 'plugin-config-wrapper'],
      ];
    }

    return $element;
  }

  /**
   * Injects a plugin selector with AJAX-enabled configuration forms.
   *
   * This method adds a checkbox-based plugin selector to the provided form
   * element. When plugins are selected, their configuration forms are
   * dynamically displayed via AJAX.
   *
   * @param array $element
   *   The form element to inject the plugin selector into.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param array $existing_configuration
   *   Existing plugin configurations, expected format:
   *   [
   *     ['plugin_id' => 'id1', 'configuration' => [...]],
   *     ['plugin_id' => 'id2', 'configuration' => [...]],
   *   ].
   * @param string $field_name
   *   The name of the field for plugin selection checkboxes.
   * @param int $argument_index
   *   The argument index, used for generating unique wrapper IDs.
   * @param bool $multiple_cardinality
   *   TRUE to allow multiple plugin selection (default), FALSE for single.
   */
  protected function injectPluginSelector(
    array &$element,
    FormStateInterface $form_state,
    array $existing_configuration,
    string $field_name,
    int $argument_index,
    bool $multiple_cardinality = TRUE,
  ): void {
    $plugin_manager = $this->completionProviderManager();

    // Get all available plugin definitions.
    $definitions = $plugin_manager->getDefinitions();

    // Build options array for checkboxes.
    $plugin_options = array_map(
      static fn(array $definition): string => (string) ($definition['label'] ?? $definition['id']),
      $definitions
    );

    // Determine which plugins are currently selected.
    // During AJAX/form submission, use form state values (even if empty).
    // Only fall back to existing settings on initial form load.
    $triggering_element = $form_state->getTriggeringElement();
    $is_ajax_or_submitted = $triggering_element || $form_state->isSubmitted();

    if ($is_ajax_or_submitted) {
      $selected_plugins = $this->getSelectedPluginsFromFormState($form_state, $field_name, $argument_index);
    }
    else {
      $selected_plugins = $this->getSelectedPluginsFromFormState($form_state, $field_name, $argument_index);
      // Only use existing config if no form values found.
      if (empty($selected_plugins)) {
        $selected_plugins = $this->getSelectedPluginsFromSettings($existing_configuration);
      }
    }

    $wrapper_id = "argument-{$argument_index}-completion-config-wrapper";

    // Add the plugin selector (checkboxes or radios).
    $element[$field_name] = [
      '#type' => $multiple_cardinality ? 'checkboxes' : 'radios',
      '#title' => $this->t('Completion providers'),
      '#description' => $this->t('Select which completion providers to use for this argument.'),
      '#options' => $plugin_options,
      '#default_value' => $multiple_cardinality ? $selected_plugins : ($selected_plugins[0] ?? NULL),
      '#ajax' => [
        'callback' => [$this, 'pluginSelectionAjaxCallback'],
        'wrapper' => $wrapper_id,
        'event' => 'change',
      ],
    ];

    // Add the configuration wrapper for plugin configuration forms.
    $element['config_wrapper'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $wrapper_id,
      ],
    ];

    // Build configuration forms for each selected plugin.
    foreach ($selected_plugins as $plugin_id) {
      if (!$plugin_id || !isset($definitions[$plugin_id])) {
        continue;
      }

      // Find existing configuration for this plugin.
      $plugin_configuration = [];
      foreach ($existing_configuration as $config) {
        if (($config['plugin_id'] ?? NULL) === $plugin_id) {
          $plugin_configuration = $config['configuration'] ?? [];
          break;
        }
      }

      // Create plugin instance with existing configuration.
      $plugin = $plugin_manager->createInstance($plugin_id, $plugin_configuration);

      if (!$plugin instanceof PluginFormInterface) {
        continue;
      }

      // Build the plugin's configuration form.
      $element['config_wrapper'][$plugin_id] = $this->buildPluginConfigurationForm(
        $plugin,
        $element['config_wrapper'],
        $element,
        $form_state,
        (string) ($definitions[$plugin_id]['label'] ?? $plugin_id)
      );
    }
  }

  /**
   * Builds a plugin configuration form.
   *
   * @param \Drupal\Core\Plugin\PluginFormInterface $plugin
   *   The plugin instance.
   * @param array $config_wrapper
   *   The configuration wrapper element.
   * @param array $parent_element
   *   The parent form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param string $label
   *   The plugin label for the fieldset title.
   *
   * @return array
   *   The plugin configuration form wrapped in a details element.
   */
  protected function buildPluginConfigurationForm(
    PluginFormInterface $plugin,
    array $config_wrapper,
    array $parent_element,
    FormStateInterface $form_state,
    string $label,
  ): array {
    $subform = [];
    $subform_state = SubformState::createForSubform($config_wrapper, $parent_element, $form_state);
    $plugin_form = $plugin->buildConfigurationForm($subform, $subform_state);

    $form = [
      '#type' => 'details',
      '#title' => $label,
      '#open' => TRUE,
    ];

    if (empty($plugin_form)) {
      $form['#markup'] = $this->t('- No configuration options available for this plugin -');
    }
    else {
      $form += $plugin_form;
    }

    return $form;
  }

  /**
   * Validates plugin configuration forms.
   *
   * This method should be called from the form's validation handler for each
   * argument that uses plugin selection.
   *
   * @param array $form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param string $field_name
   *   The name of the plugin selection field.
   * @param int $argument_index
   *   The argument index.
   * @param bool $multiple_cardinality
   *   TRUE for multiple selection, FALSE for single selection.
   */
  protected function validatePluginForm(
    array &$form,
    FormStateInterface $form_state,
    string $field_name,
    int $argument_index,
    bool $multiple_cardinality = TRUE,
  ): void {
    $selected_plugins = $this->getSelectedPluginsFromFormState(
      $form_state,
      $field_name,
      $argument_index
    );

    if (empty($selected_plugins)) {
      return;
    }

    $plugin_manager = $this->completionProviderManager();

    foreach ($selected_plugins as $plugin_id) {
      if (!$plugin_id) {
        continue;
      }

      // Verify the configuration form element exists.
      $element_parents = ['arguments', $argument_index, 'config_wrapper', $plugin_id];
      $element = NestedArray::getValue($form, $element_parents);

      if (!$element) {
        continue;
      }

      // Create plugin instance and validate its configuration.
      $plugin = $plugin_manager->createInstance($plugin_id, []);

      if (!$plugin instanceof PluginFormInterface) {
        continue;
      }

      $parent_element = NestedArray::getValue($form, ['arguments', $argument_index]);
      $config_wrapper = NestedArray::getValue($form, ['arguments', $argument_index, 'config_wrapper']);
      $subform_state = SubformState::createForSubform($config_wrapper, $parent_element, $form_state);

      $plugin->validateConfigurationForm($element, $subform_state);
    }
  }

  /**
   * Processes plugin configuration on form submission.
   *
   * This method should be called from the form's submission handler for each
   * argument that uses plugin selection. It returns an array of plugin
   * configurations ready to be saved.
   *
   * @param array $form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param string $field_name
   *   The name of the plugin selection field.
   * @param int $argument_index
   *   The argument index.
   * @param bool $multiple_cardinality
   *   TRUE for multiple selection, FALSE for single selection.
   *
   * @return array
   *   Array of plugin configurations, format:
   *   [
   *     ['plugin_id' => 'id1', 'configuration' => [...]],
   *     ['plugin_id' => 'id2', 'configuration' => [...]],
   *   ].
   */
  protected function updatePluginConfiguration(
    array &$form,
    FormStateInterface $form_state,
    string $field_name,
    int $argument_index,
    bool $multiple_cardinality = TRUE,
  ): array {
    $selected_plugins = $this->getSelectedPluginsFromFormState(
      $form_state,
      $field_name,
      $argument_index
    );

    if (empty($selected_plugins)) {
      return [];
    }

    $configurations = [];
    $plugin_manager = $this->completionProviderManager();

    foreach ($selected_plugins as $plugin_id) {
      if (!$plugin_id) {
        continue;
      }

      // Verify the configuration form element exists.
      $element_parents = ['arguments', $argument_index, 'config_wrapper', $plugin_id];
      $element = NestedArray::getValue($form, $element_parents);

      if (!$element) {
        // Plugin has no configuration form, store with empty configuration.
        $configurations[] = [
          'plugin_id' => $plugin_id,
          'configuration' => [],
        ];
        continue;
      }

      // Get plugin values from form state for this specific plugin's form.
      $plugin_values = $form_state->getValue([
        'arguments',
        $argument_index,
        'config_wrapper',
        $plugin_id,
      ]) ?? [];

      // Create plugin instance with the submitted values.
      $plugin = $plugin_manager->createInstance($plugin_id, $plugin_values);

      if (!$plugin instanceof PluginFormInterface) {
        $configurations[] = [
          'plugin_id' => $plugin_id,
          'configuration' => [],
        ];
        continue;
      }

      // Create SubformState using the specific plugin's form element
      // (not the wrapper).
      $subform_state = SubformState::createForSubform($element, $form, $form_state);
      $plugin->submitConfigurationForm($element, $subform_state);

      // Store the plugin ID and its configuration.
      $configuration = [];
      if ($plugin instanceof ConfigurableInterface) {
        $configuration = $plugin->getConfiguration();
      }
      $configurations[] = [
        'plugin_id' => $plugin_id,
        'configuration' => $configuration,
      ];
    }

    return $configurations;
  }

  /**
   * Gets selected plugin IDs from form state.
   *
   * This handles both AJAX requests (from triggering element) and regular
   * form submissions (from form values).
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param string $field_name
   *   The name of the plugin selection field.
   * @param int $argument_index
   *   The argument index.
   *
   * @return array
   *   Array of selected plugin IDs with empty values filtered out.
   */
  protected function getSelectedPluginsFromFormState(
    FormStateInterface $form_state,
    string $field_name,
    int $argument_index,
  ): array {
    $triggering_element = $form_state->getTriggeringElement();

    // Check if this is an AJAX request from the plugin selector.
    // For checkboxes, the parents include the checkbox value, so we check
    // if the field_name is in the parents path (not just the last element).
    if ($triggering_element && isset($triggering_element['#parents'])) {
      if (in_array($field_name, $triggering_element['#parents'], TRUE)) {
        // Get the value directly from user input during AJAX rebuild.
        $user_input = $form_state->getUserInput();
        $plugin_values = $user_input['arguments'][$argument_index][$field_name] ?? [];
        return is_array($plugin_values) ? array_filter($plugin_values) : [];
      }
    }

    // Try to get from form values if submitted.
    if ($form_state->hasValue(['arguments', $argument_index, $field_name])) {
      $plugin_values = $form_state->getValue([
        'arguments',
        $argument_index,
        $field_name,
      ]);
      return is_array($plugin_values) ? array_filter($plugin_values) : [];
    }

    return [];
  }

  /**
   * Gets selected plugin IDs from existing configuration.
   *
   * @param array $existing_configuration
   *   Array of plugin configurations.
   *
   * @return array
   *   Array of plugin IDs.
   */
  protected function getSelectedPluginsFromSettings(array $existing_configuration): array {
    return array_filter(
      array_map(
        static fn($config): ?string => $config['plugin_id'] ?? NULL,
        $existing_configuration
      )
    );
  }

  /**
   * Gets the completion provider manager.
   *
   * Implementing classes must provide this method.
   *
   * @return \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager
   *   The plugin manager instance.
   */
  abstract protected function completionProviderManager(): PromptArgumentCompletionProviderManager;

}
