<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\mcp_server\Plugin\ResourceTemplateManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a unified form for configuring resource template plugins.
 *
 * This form displays all available resource template plugins with checkboxes
 * to enable them and allows each plugin to provide its own configuration
 * form elements.
 */
final class ResourcePluginSettingsForm extends ConfigFormBase {

  /**
   * Constructs a ResourcePluginSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager service.
   * @param \Drupal\mcp_server\Plugin\ResourceTemplateManager $resourceTemplateManager
   *   The resource template plugin manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    private readonly ResourceTemplateManager $resourceTemplateManager,
    private readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.mcp_server.resource_template'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['mcp_server.resource_plugins'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'mcp_server_resource_plugin_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('mcp_server.resource_plugins');
    $saved_plugins = $config->get('plugins') ?? [];

    // Build an associative array keyed by plugin ID for easier lookup.
    $plugin_configs = [];
    foreach ($saved_plugins as $plugin_config) {
      $plugin_configs[$plugin_config['id']] = $plugin_config;
    }

    $form['plugins'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    $definitions = $this->resourceTemplateManager->getAvailableDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $plugin_config = $plugin_configs[$plugin_id] ?? [
        'id' => $plugin_id,
        'enabled' => FALSE,
        'configuration' => [],
      ];

      // Create plugin instance with current configuration.
      $plugin = $this->resourceTemplateManager->createInstance(
        $plugin_id,
        $plugin_config['configuration'] ?? []
      );

      $form['plugins'][$plugin_id] = [
        '#type' => 'details',
        '#title' => $definition['label'],
        '#open' => $plugin_config['enabled'] ?? FALSE,
      ];

      if (!empty($definition['description'])) {
        $form['plugins'][$plugin_id]['description'] = [
          '#type' => 'markup',
          '#markup' => '<p>' . $definition['description'] . '</p>',
        ];
      }

      $form['plugins'][$plugin_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable this resource plugin'),
        '#default_value' => $plugin_config['enabled'] ?? FALSE,
      ];

      // Check for module dependencies.
      $dependencies = $definition['module_dependencies'] ?? [];
      if (!empty($dependencies)) {
        $missing_deps = [];
        foreach ($dependencies as $module) {
          if (!$this->moduleHandler->moduleExists($module)) {
            $missing_deps[] = $module;
          }
        }

        if (!empty($missing_deps)) {
          $form['plugins'][$plugin_id]['enabled']['#disabled'] = TRUE;
          $form['plugins'][$plugin_id]['dependencies_warning'] = [
            '#type' => 'markup',
            '#markup' => '<div class="messages messages--warning">' . $this->t('This plugin requires the following modules: @modules', [
              '@modules' => implode(', ', $missing_deps),
            ]) . '</div>',
          ];
        }
      }

      // Allow plugin to add its own configuration form elements.
      $form['plugins'][$plugin_id]['configuration'] = [
        '#type' => 'container',
        '#tree' => TRUE,
        '#states' => [
          'visible' => [
            ':input[name="plugins[' . $plugin_id . '][enabled]"]' => [
              'checked' => TRUE,
            ],
          ],
        ],
      ];

      // Create a subform state for the plugin configuration.
      $subform_state = SubformState::createForSubform(
        $form['plugins'][$plugin_id]['configuration'],
        $form,
        $form_state
      );

      $plugin_form = $plugin->buildConfigurationForm(
        $form['plugins'][$plugin_id]['configuration'],
        $subform_state
      );
      $form['plugins'][$plugin_id]['configuration'] = $plugin_form;
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $plugins_values = $form_state->getValue('plugins');
    if (!is_array($plugins_values)) {
      return;
    }

    foreach ($plugins_values as $plugin_id => $values) {
      if (empty($values['enabled'])) {
        continue;
      }

      // Create plugin instance and call its validation method.
      try {
        $plugin = $this->resourceTemplateManager->createInstance(
          $plugin_id,
          $values['configuration'] ?? []
        );

        // Create a subform state for the plugin configuration.
        $subform_state = SubformState::createForSubform(
          $form['plugins'][$plugin_id]['configuration'],
          $form,
          $form_state
        );

        $plugin->validateConfigurationForm($form['plugins'][$plugin_id]['configuration'], $subform_state);
      }
      catch (\Exception $e) {
        $form_state->setErrorByName(
          "plugins][$plugin_id]",
          $this->t('Error validating plugin @plugin: @message', [
            '@plugin' => $plugin_id,
            '@message' => $e->getMessage(),
          ])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $plugins_values = $form_state->getValue('plugins');
    if (!is_array($plugins_values)) {
      return;
    }

    $plugins_config = [];

    foreach ($plugins_values as $plugin_id => $values) {
      // Create plugin instance and call its submit method.
      try {
        $plugin = $this->resourceTemplateManager->createInstance(
          $plugin_id,
          $values['configuration'] ?? []
        );

        // Create a subform state for the plugin configuration.
        $subform_state = SubformState::createForSubform(
          $form['plugins'][$plugin_id]['configuration'],
          $form,
          $form_state
        );

        $plugin->submitConfigurationForm($form['plugins'][$plugin_id]['configuration'], $subform_state);

        // Get the processed configuration from the plugin.
        $configuration = $plugin->getConfiguration();

        $plugins_config[] = [
          'id' => $plugin_id,
          'enabled' => (bool) ($values['enabled'] ?? FALSE),
          'configuration' => $configuration,
        ];
      }
      catch (\Exception $e) {
        $this->messenger()->addError(
          $this->t('Error processing plugin @plugin: @message', [
            '@plugin' => $plugin_id,
            '@message' => $e->getMessage(),
          ])
        );
      }
    }

    $this->config('mcp_server.resource_plugins')
      ->set('plugins', $plugins_config)
      ->save();

    // Invalidate the discovery cache.
    Cache::invalidateTags(['mcp_server:discovery']);

    parent::submitForm($form, $form_state);
  }

}
