<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Handler;

use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Drupal\mcp_server\Exception\InsufficientScopeException;
use Drupal\mcp_server\McpBridgeService;
use Mcp\Schema\Content\TextContent;
use Mcp\Server\ClientGateway;
use Mcp\Schema\JsonRpc\Error;
use Mcp\Schema\JsonRpc\Request;
use Mcp\Schema\JsonRpc\Response;
use Mcp\Schema\Request\CallToolRequest;
use Mcp\Schema\Result\CallToolResult;
use Mcp\Server\Handler\Request\RequestHandlerInterface;
use Mcp\Server\Session\SessionInterface;
use Psr\Log\LoggerInterface;

/**
 * Custom tool call handler that bypasses SDK's parameter name matching.
 *
 * This handler receives tool call requests and passes arguments directly
 * to the bridge service as an associative array, avoiding the need for
 * reflection-based parameter mapping or code generation.
 *
 * @implements RequestHandlerInterface<CallToolResult>
 */
final class CustomCallToolHandler implements RequestHandlerInterface {

  /**
   * Constructs a CustomCallToolHandler.
   *
   * @param \Drupal\mcp_server\McpBridgeService $bridge
   *   The MCP bridge service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    private readonly McpBridgeService $bridge,
    private readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function supports(Request $request): bool {
    return $request instanceof CallToolRequest;
  }

  /**
   * {@inheritdoc}
   */
  public function handle(Request $request, SessionInterface $session): Response|Error {
    assert($request instanceof CallToolRequest);

    $tool_name = $request->name;
    $arguments = $request->arguments ?? [];

    // Create ClientGateway from session for tools that support sampling.
    $gateway = new ClientGateway($session);

    try {
      // Call bridge service directly with arguments as associative array.
      // No parameter name matching or code generation needed!
      $result = $this->bridge->executeMcpTool($tool_name, $arguments, $gateway);

      // Wrap result in CallToolResult if needed.
      if (!$result instanceof CallToolResult) {
        // Format the result as text content.
        $formatted = $this->formatResult($result);
        $result = new CallToolResult($formatted);
      }

      return new Response($request->getId(), $result);
    }
    catch (AuthenticationRequiredException $e) {
      $this->logger->info('Authentication required for tool', [
        'tool' => $tool_name,
      ]);

      // Return proper JSON-RPC error with marker for controller.
      // Include structured data so controller can extract tool info.
      return new Error(
        id: $request->getId(),
        code: -32603,
        message: '[AUTH_REQUIRED]' . $e->getMessage(),
        data: [
          'tool' => $e->getToolName(),
          'authentication_mode' => $e->getAuthenticationMode(),
        ],
      );
    }
    catch (InsufficientScopeException $e) {
      $this->logger->info('Insufficient scopes for tool', [
        'tool' => $tool_name,
      ]);

      // Return proper JSON-RPC error with marker for controller.
      // Include structured data so controller can extract error info.
      return new Error(
        id: $request->getId(),
        code: -32603,
        message: '[INSUFFICIENT_SCOPE]' . $e->getMessage(),
        data: $e->getErrorData(),
      );
    }
    catch (\Throwable $e) {
      $this->logger->error('Tool execution failed', [
        'tool' => $tool_name,
        'message' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
      ]);

      $content = [new TextContent('Tool execution failed: ' . $e->getMessage())];
      return new Response($request->getId(), CallToolResult::error($content));
    }
  }

  /**
   * Formats tool result for MCP response.
   *
   * @param mixed $result
   *   The tool execution result.
   *
   * @return array
   *   Formatted content array.
   */
  private function formatResult(mixed $result): array {
    if (is_array($result)) {
      // Convert array to JSON for display.
      $json = json_encode($result, JSON_PRETTY_PRINT);
      return [new TextContent($json)];
    }

    if (is_string($result)) {
      return [new TextContent($result)];
    }

    if (is_scalar($result)) {
      return [new TextContent((string) $result)];
    }

    // For objects, try to serialize.
    return [new TextContent(print_r($result, TRUE))];
  }

}
