<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\mcp_server\Entity\McpPromptConfig;
use Drupal\mcp_server\Entity\McpToolConfig;
use Drupal\mcp_server\Exception\AuthenticationRequiredException;
use Drupal\mcp_server\Exception\InsufficientScopeException;
use Drupal\mcp_server\Service\OAuthScopeValidator;
use Mcp\Server\ClientGateway;

/**
 * Bridge service integrating Tool API with MCP configurations.
 *
 * This service provides a unified interface for managing MCP tools and
 * prompts by combining Tool API discovery with configuration entity overrides.
 */
final class McpBridgeService {

  /**
   * Constructs an McpBridgeService.
   *
   * @param \Drupal\mcp_server\ToolApiDiscovery $toolApiDiscovery
   *   The Tool API discovery service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\mcp_server\Service\OAuthScopeValidator $scopeValidator
   *   The OAuth scope validator service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   */
  public function __construct(
    private readonly ToolApiDiscovery $toolApiDiscovery,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerChannelInterface $logger,
    private readonly OAuthScopeValidator $scopeValidator,
    private readonly AccountProxyInterface $currentUser,
  ) {}

  /**
   * Gets all enabled MCP tools.
   *
   * Returns tools that have configuration entities marked as enabled.
   * If no configuration exists for a tool, it's considered disabled.
   *
   * @return array<string, array>
   *   Array of enabled MCP tools keyed by MCP tool name. Each tool contains:
   *   - id: string - The MCP tool name
   *   - label: string - Human-readable label
   *   - description: string - Tool description
   *   - tool_id: string - Original Tool API plugin ID
   *   - inputSchema: array - JSON Schema for parameters
   *   - destructive: bool - Whether the tool is destructive
   */
  public function getEnabledTools(): array {
    try {
      $storage = $this->entityTypeManager->getStorage('mcp_tool_config');
      $configs = $storage->loadMultiple();
      $enabled_tools = [];

      foreach ($configs as $config) {
        if (!$config instanceof McpToolConfig) {
          continue;
        }

        if (!$config->status()) {
          continue;
        }

        $tool_id = $config->getToolId();
        $tool_definition = $this->toolApiDiscovery->getToolDefinition($tool_id);

        if ($tool_definition === NULL) {
          $this->logger->warning(
            'Tool @tool_id referenced in config @config_id not found',
            [
              '@tool_id' => $tool_id,
              '@config_id' => $config->id(),
            ]
          );
          continue;
        }

        // Apply configuration overrides.
        // Use the entity ID as the MCP tool name since it's a machine name
        // that conforms to MCP requirements (alphanumeric + underscore).
        $mcp_name = $config->id();
        $tool_metadata = [
          'id' => $mcp_name,
          'label' => $config->label(),
          'description' => $config->getDescription() ?? $tool_definition['description'],
          'tool_id' => $tool_id,
          'inputSchema' => $tool_definition['inputSchema'],
          'destructive' => $tool_definition['destructive'],
        ];

        // Add authorization metadata if authentication is configured.
        if ($config->requiresAuthentication()) {
          $tool_metadata['authorization'] = [
            'method' => 'oauth2',
            'mode' => $config->getAuthenticationMode(),
            'requiredScopes' => $config->getScopeNames(),
          ];
        }

        $enabled_tools[$mcp_name] = $tool_metadata;
      }

      return $enabled_tools;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve enabled MCP tools: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

  /**
   * Gets a specific MCP tool by its MCP name.
   *
   * @param string $mcpName
   *   The MCP tool name.
   *
   * @return array|null
   *   Tool definition array with applied configuration overrides, or NULL
   *   if the tool is not found or disabled.
   */
  public function getMcpTool(string $mcpName): ?array {
    try {
      $storage = $this->entityTypeManager->getStorage('mcp_tool_config');

      // The MCP name is the entity ID, so we can load it directly.
      $config = $storage->load($mcpName);

      if (!$config instanceof McpToolConfig || !$config->status()) {
        return NULL;
      }

      $tool_id = $config->getToolId();
      $tool_definition = $this->toolApiDiscovery->getToolDefinition($tool_id);

      if ($tool_definition === NULL) {
        $this->logger->warning(
          'Tool @tool_id not found for MCP name @mcp_name',
          [
            '@tool_id' => $tool_id,
            '@mcp_name' => $mcpName,
          ]
        );
        return NULL;
      }

      // Apply configuration overrides.
      $tool_metadata = [
        'id' => $mcpName,
        'label' => $config->label(),
        'description' => $config->getDescription() ?? $tool_definition['description'],
        'tool_id' => $tool_id,
        'inputSchema' => $tool_definition['inputSchema'],
        'destructive' => $tool_definition['destructive'],
      ];

      // Add authorization metadata if authentication is configured.
      if ($config->requiresAuthentication()) {
        $tool_metadata['authorization'] = [
          'method' => 'oauth2',
          'mode' => $config->getAuthenticationMode(),
          'requiredScopes' => $config->getScopeNames(),
        ];
      }

      return $tool_metadata;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve MCP tool @mcp_name: @message',
        [
          '@mcp_name' => $mcpName,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Executes an MCP tool by its MCP name.
   *
   * @param string $mcpName
   *   The MCP tool name.
   * @param array $parameters
   *   Tool parameters keyed by input name.
   * @param \Mcp\Server\ClientGateway|null $gateway
   *   Optional client gateway for tools that support sampling.
   *
   * @return mixed
   *   Tool execution result.
   *
   * @throws \RuntimeException
   *   If the tool is not found, disabled, or execution fails.
   * @throws \Drupal\mcp_server\Exception\AuthenticationRequiredException
   *   If authentication is required but user is anonymous.
   * @throws \Drupal\mcp_server\Exception\InsufficientScopeException
   *   If required scopes are missing in required authentication mode.
   */
  public function executeMcpTool(
    string $mcpName,
    array $parameters,
    ?ClientGateway $gateway = NULL,
  ): mixed {
    // Load tool configuration to check authentication requirements.
    $tool_config = $this->loadToolConfigByMcpName($mcpName);

    if ($tool_config === NULL) {
      $this->logger->error('Tool config not found for MCP name: @name', ['@name' => $mcpName]);
      throw new \RuntimeException(
        "MCP tool '$mcpName' not found or is disabled"
      );
    }

    $authentication_mode = $tool_config->getAuthenticationMode();
    $is_anonymous = $this->currentUser->isAnonymous();

    // Fast path: if authentication is disabled, skip all checks.
    if ($tool_config->getAuthenticationMode() === 'disabled') {
      return $this->executeToolById($tool_config->getToolId(), $parameters, $gateway);
    }

    // Handle required authentication mode.
    if ($authentication_mode === 'required') {
      // Check authentication requirement.
      if ($is_anonymous) {
        throw new AuthenticationRequiredException($mcpName, $authentication_mode);
      }

      // Validate OAuth scopes (throws exception on failure).
      $this->validateToolScopes($tool_config, $mcpName);

      return $this->executeToolById($tool_config->getToolId(), $parameters, $gateway);
    }

    // Fallback for unexpected authentication mode.
    return $this->executeToolById($tool_config->getToolId(), $parameters, $gateway);
  }

  /**
   * Loads a tool configuration by its MCP name.
   *
   * @param string $mcpName
   *   The MCP tool name.
   *
   * @return \Drupal\mcp_server\Entity\McpToolConfig|null
   *   The tool configuration entity, or NULL if not found or disabled.
   */
  private function loadToolConfigByMcpName(string $mcpName): ?McpToolConfig {
    try {
      $storage = $this->entityTypeManager->getStorage('mcp_tool_config');

      // The MCP name is the entity ID, so we can load it directly.
      $config = $storage->load($mcpName);

      if (!$config instanceof McpToolConfig || !$config->status()) {
        return NULL;
      }

      return $config;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to load tool config for @mcp_name: @message',
        [
          '@mcp_name' => $mcpName,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Executes a Tool API tool by its ID.
   *
   * @param string $toolId
   *   The Tool API tool ID.
   * @param array $parameters
   *   Tool parameters keyed by input name.
   * @param \Mcp\Server\ClientGateway|null $gateway
   *   Optional client gateway for tools that support sampling.
   *
   * @return mixed
   *   Tool execution result.
   *
   * @throws \RuntimeException
   *   If tool execution fails.
   */
  private function executeToolById(
    string $toolId,
    array $parameters,
    ?ClientGateway $gateway = NULL,
  ): mixed {
    try {
      return $this->toolApiDiscovery->executeTool($toolId, $parameters, $gateway);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to execute tool @tool_id: @message',
        [
          '@tool_id' => $toolId,
          '@message' => $e->getMessage(),
        ]
      );
      throw new \RuntimeException(
        "Failed to execute tool '$toolId': {$e->getMessage()}",
        0,
        $e
      );
    }
  }

  /**
   * Gets all available Tool API tools for configuration.
   *
   * This returns all tools discovered by the Tool API, regardless of whether
   * they have MCP configuration entities. Useful for admin UI to show
   * available tools that can be configured.
   *
   * @return array<string, array>
   *   Array of all available Tool API tools keyed by tool ID.
   */
  public function getAvailableToolApiTools(): array {
    return $this->toolApiDiscovery->getAvailableTools();
  }

  /**
   * Checks if an MCP tool is enabled.
   *
   * @param string $mcpName
   *   The MCP tool name.
   *
   * @return bool
   *   TRUE if the tool is enabled, FALSE otherwise.
   */
  public function isToolEnabled(string $mcpName): bool {
    return $this->getMcpTool($mcpName) !== NULL;
  }

  /**
   * Gets the Tool API tool ID for an MCP tool name.
   *
   * @param string $mcpName
   *   The MCP tool name.
   *
   * @return string|null
   *   The Tool API tool ID, or NULL if not found.
   */
  public function getToolApiId(string $mcpName): ?string {
    $tool = $this->getMcpTool($mcpName);
    return $tool['tool_id'] ?? NULL;
  }

  /**
   * Validates OAuth scopes for a tool configuration.
   *
   * This method validates that the current request's OAuth token contains all
   * required scopes for the given tool. If scopes are empty, any authenticated
   * user is allowed.
   *
   * @param \Drupal\mcp_server\Entity\McpToolConfig $config
   *   The tool configuration entity.
   * @param string $mcpName
   *   The MCP tool name for error messages.
   *
   * @throws \Drupal\mcp_server\Exception\InsufficientScopeException
   *   If required scopes are missing.
   */
  protected function validateToolScopes(
    McpToolConfig $config,
    string $mcpName,
  ): void {
    $required_scopes = $config->getScopeNames();

    // No restrictions if scopes array is empty.
    if (empty($required_scopes)) {
      return;
    }

    // Extract scopes from the current request's OAuth token.
    $token_scopes = $this->scopeValidator->extractTokenScopes();

    // Validate that all required scopes are present.
    $validation = $this->scopeValidator->validateScopes(
      $required_scopes,
      $token_scopes
    );

    if (!$validation->isValid) {
      $message = sprintf(
        'User lacks required scopes for tool %s. Required: [%s], Missing: [%s], Current: [%s]',
        $mcpName,
        implode(', ', $validation->requiredScopes),
        implode(', ', $validation->missingScopes),
        implode(', ', $validation->tokenScopes)
      );

      throw new InsufficientScopeException(
        $message,
        $mcpName,
        $validation->requiredScopes,
        $validation->missingScopes,
        $validation->tokenScopes
      );
    }
  }

  /**
   * Gets all enabled MCP prompts.
   *
   * Returns prompts that have configuration entities marked as enabled.
   * Prompts are discoverable by all users; permission checks are enforced
   * at execution time via getMcpPrompt().
   *
   * @return array<string, array>
   *   Array of enabled MCP prompts keyed by prompt name. Each prompt contains:
   *   - name: string - The prompt identifier
   *   - title: string|null - Optional human-readable title
   *   - description: string|null - Optional prompt description
   *   - arguments: array - Array of argument definitions
   *   - messages: array - Array of message objects with role and content
   */
  public function getEnabledPrompts(): array {

    try {
      $storage = $this->entityTypeManager->getStorage('mcp_prompt_config');
      $query = $storage->getQuery()
        ->condition('status', TRUE)
        ->accessCheck(FALSE);

      $config_ids = $query->execute();
      $configs = $storage->loadMultiple($config_ids);
      $enabled_prompts = [];

      foreach ($configs as $config) {
        if (!$config instanceof McpPromptConfig) {
          continue;
        }

        // Use the entity ID as the MCP prompt name since it's a machine name
        // that conforms to MCP requirements (alphanumeric + underscore).
        $prompt_name = $config->id();
        $enabled_prompts[$prompt_name] = [
          'name' => $prompt_name,
          'title' => $config->getTitle(),
          'description' => $config->getDescription(),
          'arguments' => $config->getArguments(),
          'messages' => $config->getMessages(),
        ];
      }

      return $enabled_prompts;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve enabled MCP prompts: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

  /**
   * Gets a specific MCP prompt by its name.
   *
   * @param string $name
   *   The prompt name.
   *
   * @return array|null
   *   Prompt definition array in MCP protocol format, or NULL if the prompt
   *   is not found, disabled, or user lacks permission.
   */
  public function getMcpPrompt(string $name): ?array {
    if (!$this->currentUser->hasPermission('access mcp server prompts')) {
      return NULL;
    }

    try {
      $storage = $this->entityTypeManager->getStorage('mcp_prompt_config');

      // The MCP name is the entity ID, so we can load it directly.
      $config = $storage->load($name);

      if (!$config instanceof McpPromptConfig || !$config->status()) {
        return NULL;
      }

      return [
        'name' => $name,
        'title' => $config->getTitle(),
        'description' => $config->getDescription(),
        'arguments' => $config->getArguments(),
        'messages' => $config->getMessages(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve MCP prompt @name: @message',
        [
          '@name' => $name,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

}
