<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\mcp_server\Plugin\ResourceTemplateInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\mcp_server\Plugin\ResourceTemplateManager;

/**
 * Bridge service integrating Resource plugins with MCP configurations.
 *
 * This service provides a unified interface for managing MCP resources by
 * combining resource template plugins with configuration entity overrides.
 * It discovers enabled resource configurations, manages plugin instances,
 * and handles resource content retrieval and access control.
 */
final class McpResourceBridgeService {

  /**
   * Constructs a McpResourceBridgeService.
   *
   * @param \Drupal\mcp_server\Plugin\ResourceTemplateManager $resourceTemplateManager
   *   The resource template plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend service.
   */
  public function __construct(
    private readonly ResourceTemplateManager $resourceTemplateManager,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerChannelInterface $logger,
    private readonly AccountProxyInterface $currentUser,
    private readonly CacheBackendInterface $cache,
  ) {}

  /**
   * Gets all enabled MCP resources.
   *
   * Returns resources from all enabled resource plugin configurations. This
   * method reads plugin configurations from the mcp_server.resource_plugins
   * config object and instantiates enabled plugins to discover available
   * resources. Results are cached to avoid repeated plugin instantiation.
   *
   * @return array<int, array>
   *   Array of resource metadata. Each resource contains:
   *   - uri: string - The resource URI template
   *   - name: string - Human-readable name
   *   - description: string - Resource description
   *   - mimeType: string - MIME type
   */
  public function getEnabledResources(): array {
    $cache_id = 'mcp_server:enabled_resources';
    $cached = $this->cache->get($cache_id);

    if ($cached !== FALSE) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('mcp_server.resource_plugins');
      $plugins = $config->get('plugins') ?? [];
      $enabled_resources = [];

      foreach ($plugins as $plugin_config) {
        if (empty($plugin_config['enabled'])) {
          continue;
        }

        $plugin_id = $plugin_config['id'];

        try {
          $plugin = $this->resourceTemplateManager->createInstance(
            $plugin_id,
            $plugin_config['configuration'] ?? []
          );

          $resources = $plugin->getResources();

          foreach ($resources as $resource) {
            $enabled_resources[] = $resource;
          }
        }
        catch (\Exception $e) {
          $this->logger->error(
            'Failed to load resource plugin @plugin_id: @message',
            [
              '@plugin_id' => $plugin_id,
              '@message' => $e->getMessage(),
            ]
          );
        }
      }

      $this->cache->set(
        $cache_id,
        $enabled_resources,
        CacheBackendInterface::CACHE_PERMANENT,
        ['mcp_server:discovery']
      );

      return $enabled_resources;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to retrieve enabled resources: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

  /**
   * Gets content for a specific resource URI.
   *
   * Finds the appropriate resource template plugin that handles the given URI
   * and delegates content retrieval to it. Access is checked before returning
   * content.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return array|null
   *   Resource content array, or NULL if not found/accessible.
   */
  public function getResourceContent(string $uri): ?array {
    // Find which plugin handles this URI.
    $plugin = $this->findPluginForUri($uri);
    if ($plugin === NULL) {
      return NULL;
    }

    // Check access first.
    $access = $plugin->checkAccess($uri, $this->currentUser->getAccount());
    if (!$access->isAllowed()) {
      return NULL;
    }

    // Get content from plugin.
    try {
      return $plugin->getResourceContent($uri);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to get resource content for @uri: @message',
        [
          '@uri' => $uri,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Checks access for a specific resource URI.
   *
   * Finds the appropriate resource template plugin that handles the given URI
   * and delegates access checking to it.
   *
   * @param string $uri
   *   The resource URI.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result with cacheability metadata.
   */
  public function checkResourceAccess(
    string $uri,
    AccountInterface $account,
  ): AccessResultInterface {
    $plugin = $this->findPluginForUri($uri);
    if ($plugin === NULL) {
      return AccessResult::forbidden('No plugin found for URI');
    }

    try {
      return $plugin->checkAccess($uri, $account);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to check access for @uri: @message',
        [
          '@uri' => $uri,
          '@message' => $e->getMessage(),
        ]
      );
      return AccessResult::forbidden('Access check failed');
    }
  }

  /**
   * Finds the plugin instance that handles a given URI.
   *
   * This method iterates through enabled plugin configurations and checks
   * each plugin to determine if it can handle the URI. The matching is based
   * on URI patterns defined by the resource template plugins.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return \Drupal\mcp_server\Plugin\ResourceTemplateInterface|null
   *   Plugin instance, or NULL if not found.
   */
  private function findPluginForUri(string $uri): ?ResourceTemplateInterface {
    try {
      $config = $this->configFactory->get('mcp_server.resource_plugins');
      $plugins = $config->get('plugins') ?? [];

      foreach ($plugins as $plugin_config) {
        if (empty($plugin_config['enabled'])) {
          continue;
        }

        $plugin_id = $plugin_config['id'];

        try {
          $plugin = $this->resourceTemplateManager->createInstance(
            $plugin_id,
            $plugin_config['configuration'] ?? []
          );

          // Check if plugin's URI template matches the provided URI by
          // checking if the URI starts with the scheme pattern.
          $uri_template = $plugin->getUriTemplate();
          $scheme = $this->extractScheme($uri_template);

          if ($scheme !== NULL && str_starts_with($uri, $scheme)) {
            return $plugin;
          }
        }
        catch (\Exception $e) {
          continue;
        }
      }

      return NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Extracts the scheme portion from a URI template.
   *
   * Extracts the static prefix before the first template variable. For
   * example, 'drupal://entity/{entity_type}/{entity_id}' returns
   * 'drupal://entity/'.
   *
   * @param string $uriTemplate
   *   The URI template (e.g., 'drupal://entity/{entity_type}/{entity_id}').
   *
   * @return string|null
   *   The scheme prefix, or NULL if the template has no static prefix.
   */
  private function extractScheme(string $uriTemplate): ?string {
    // Find the position of the first template variable.
    $pos = strpos($uriTemplate, '{');

    if ($pos === FALSE) {
      // No template variables, return the entire string.
      return $uriTemplate;
    }

    if ($pos === 0) {
      // Template starts with a variable, no static prefix.
      return NULL;
    }

    // Return everything before the first template variable.
    return substr($uriTemplate, 0, $pos);
  }

}
