<?php

declare(strict_types=1);

namespace Drupal\mcp_server;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\mcp_server\Capability\Loader\PromptConfigLoader;
use Drupal\mcp_server\Handler\CustomCallToolHandler;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager;
use Mcp\Server;
use Mcp\Server\Session\SessionStoreInterface;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;

/**
 * Factory for building configured MCP Server instances.
 *
 * This factory creates fully configured MCP Server instances with custom
 * request handler that intercepts tool calls. Tools are registered for
 * discovery only; execution is handled by CustomCallToolHandler.
 */
final class McpServerFactory {

  /**
   * Constructs an McpServerFactory.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\mcp_server\McpBridgeService $mcpBridge
   *   The MCP bridge service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Psr\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   * @param \Mcp\Server\Session\SessionStoreInterface $sessionStore
   *   The session store.
   * @param \Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderManager $completionProviderManager
   *   The completion provider plugin manager.
   * @param \Drupal\mcp_server\McpResourceBridgeService $resourceBridge
   *   The MCP resource bridge service.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly McpBridgeService $mcpBridge,
    private readonly LoggerInterface $logger,
    private readonly EventDispatcherInterface $eventDispatcher,
    private readonly SessionStoreInterface $sessionStore,
    private readonly PromptArgumentCompletionProviderManager $completionProviderManager,
    private readonly McpResourceBridgeService $resourceBridge,
  ) {}

  /**
   * Creates a configured MCP Server instance.
   *
   * @return \Mcp\Server
   *   The configured MCP server.
   */
  public function create(): Server {
    // Read configuration.
    $config = $this->configFactory->get('mcp_server.settings');
    $server_name = $config->get('server_name') ?: 'Drupal MCP Server';
    $server_version = $config->get('server_version') ?: '1.0.0';
    $pagination_limit = $config->get('pagination_limit') ?: 50;

    // Validate pagination limit.
    $pagination_limit = (int) $pagination_limit;
    if ($pagination_limit < 1 || $pagination_limit > 1000) {
      $this->logger->warning(
        'Invalid pagination_limit (@limit). Using default: 50',
        ['@limit' => $pagination_limit]
      );
      $pagination_limit = 50;
    }

    $builder = Server::builder()
      ->setServerInfo($server_name, $server_version)
      ->setPaginationLimit($pagination_limit)
      ->setLogger($this->logger)
      ->setEventDispatcher($this->eventDispatcher)
      ->setSession($this->sessionStore);

    // Register custom handler to intercept tool calls.
    $customHandler = new CustomCallToolHandler($this->mcpBridge, $this->logger);
    $builder->addRequestHandler($customHandler);

    // Register tools with dummy handlers for metadata discovery only.
    $enabled_tools = $this->mcpBridge->getEnabledTools();
    foreach ($enabled_tools as $tool_data) {
      $builder->addTool(
        handler: fn() => NULL,
        name: $tool_data['id'],
        description: $tool_data['description'],
        inputSchema: $tool_data['inputSchema'],
      );
    }

    // Register prompts using custom loader that bypasses reflection.
    // This ensures argument descriptions from configuration are properly
    // exposed.
    $promptLoader = new PromptConfigLoader(
      $this->mcpBridge,
      $this->logger,
      $this->completionProviderManager
    );
    $builder->addLoaders($promptLoader);

    // Register resource templates from enabled resource configurations.
    try {
      $enabled_resources = $this->resourceBridge->getEnabledResources();
      foreach ($enabled_resources as $resource_data) {
        $builder->addResourceTemplate(
          handler: fn(string $uri) => $this->resourceBridge->getResourceContent($uri),
          uriTemplate: $resource_data['uri'],
          name: $resource_data['name'],
          description: $resource_data['description'] ?? '',
          mimeType: $resource_data['mimeType'] ?? 'application/json',
        );
      }
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to register MCP resource templates: @message',
        ['@message' => $e->getMessage()]
      );
    }

    try {
      return $builder->build();
    }
    catch (\Exception $e) {
      $this->logger->error(
        'MCP Server build failed: @message | @trace',
        [
          '@message' => $e->getMessage(),
          '@trace' => $e->getTraceAsString(),
        ]
      );
      throw $e;
    }
  }

}
