<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\PromptArgumentCompletionProvider;

use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides entity autocomplete based on entity queries.
 *
 * This completion provider queries Drupal entities and returns absolute URLs
 * to matching entities. The provider supports:
 * - Filtering by entity type (node, user, taxonomy_term, media, etc.)
 * - Optional bundle filtering (e.g., article vs page for nodes)
 * - Search by entity label (partial match) or entity ID (exact match)
 * - Access control based on current user permissions
 * - Automatic path alias resolution
 * - Results limited to 10 items for performance.
 *
 * @PromptArgumentCompletionProvider(
 *   id = "entity_query",
 *   label = @Translation("Entity query"),
 *   description = @Translation("Queries Drupal entities and returns absolute URLs with path aliases. Supports entity type and bundle filtering, with search by label or ID.")
 * )
 */
#[PromptArgumentCompletionProvider(
  id: 'entity_query',
  label: new TranslatableMarkup('Entity query'),
  description: new TranslatableMarkup('Queries Drupal entities and returns absolute URLs with path aliases. Supports entity type and bundle filtering, with search by label or ID.'),
)]
final class EntityQueryCompletionProvider extends PromptArgumentCompletionProviderBase {

  /**
   * Constructs a new EntityQueryCompletionProvider.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *   The bundle info service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'entity_type' => '',
      'bundle' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Entity type selector.
    $entity_type_options = ['' => $this->t('- Select one -')];
    $entity_types_with_bundles = [];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type instanceof ContentEntityTypeInterface) {
        $entity_type_options[$entity_type_id] = $entity_type->getLabel();

        // Check if this entity type has bundles.
        $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);
        if (count($bundles) > 1) {
          $entity_types_with_bundles[] = $entity_type_id;
        }
      }
    }

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type'),
      '#description' => $this->t('Select the entity type to query.'),
      '#options' => $entity_type_options,
      '#default_value' => $this->configuration['entity_type'] ?? '',
      '#required' => TRUE,
    ];

    // Create a bundle selector for each entity type that has bundles.
    foreach ($entity_types_with_bundles as $entity_type_id) {
      $bundles = $this->bundleInfo->getBundleInfo($entity_type_id);
      $bundle_options = ['' => $this->t('- All bundles -')];

      foreach ($bundles as $bundle_id => $bundle_info) {
        $bundle_options[$bundle_id] = $bundle_info['label'];
      }

      $entity_type_def = $this->entityTypeManager->getDefinition($entity_type_id);

      $form['bundle_' . $entity_type_id] = [
        '#type' => 'select',
        '#title' => $this->t('@entity_type bundle', [
          '@entity_type' => $entity_type_def->getLabel(),
        ]),
        '#description' => $this->t('Filter by a specific bundle, or select "All bundles" to query all @entity_type bundles.', [
          '@entity_type' => $entity_type_def->getLabel(),
        ]),
        '#options' => $bundle_options,
        '#default_value' => ($this->configuration['entity_type'] ?? '') === $entity_type_id
          ? ($this->configuration['bundle'] ?? '')
          : '',
        '#states' => [
          'visible' => [
            // Use name$= to match the end of the name attribute.
            // This works in both standalone and embedded subform contexts.
            ':input[name$="[entity_type]"]' => ['value' => $entity_type_id],
          ],
        ],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');

    if (empty($entity_type)) {
      $form_state->setErrorByName('entity_type', $this->t('Entity type is required.'));
      return;
    }

    // Validate that the entity type exists and is a content entity.
    $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type, FALSE);
    if (!$entity_type_definition || !($entity_type_definition instanceof ContentEntityTypeInterface)) {
      $form_state->setErrorByName('entity_type', $this->t('Invalid entity type selected.'));
      return;
    }

    // Validate bundle if specified.
    $bundle_field_name = 'bundle_' . $entity_type;
    $bundle = $form_state->getValue($bundle_field_name);

    if (!empty($bundle)) {
      // Get available bundles for this entity type.
      $bundles = $this->bundleInfo->getBundleInfo($entity_type);

      if (!isset($bundles[$bundle])) {
        $form_state->setErrorByName(
          $bundle_field_name,
          $this->t('The selected bundle "@bundle" is not valid for entity type "@entity_type".', [
            '@bundle' => $bundle,
            '@entity_type' => $entity_type_definition->getLabel(),
          ])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $entity_type = $form_state->getValue('entity_type');

    // Get the bundle value from the entity-type-specific field.
    $bundle_field_name = 'bundle_' . $entity_type;
    $bundle = $form_state->getValue($bundle_field_name) ?? '';

    // Reset configuration to only store the values we need.
    // This prevents dynamic bundle_* form fields from being saved.
    $this->configuration = [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCompletions(string $current_value, array $configuration): array {
    $entity_type = $configuration['entity_type'] ?? '';
    $bundle = $configuration['bundle'] ?? '';

    if (empty($entity_type)) {
      return [];
    }

    // Build base query with access checking based on current user.
    $query = $this->entityTypeManager
      ->getStorage($entity_type)
      ->getQuery()
      ->accessCheck(TRUE)
      ->range(0, 10);

    // Add bundle condition if specified.
    if ($bundle) {
      $bundle_key = $this->entityTypeManager
        ->getDefinition($entity_type)
        ->getKey('bundle');
      if ($bundle_key) {
        $query->condition($bundle_key, $bundle);
      }
    }

    // Add search filter (label OR ID).
    if ($current_value !== '') {
      $label_key = $this->entityTypeManager
        ->getDefinition($entity_type)
        ->getKey('label');

      $or_group = $query->orConditionGroup();

      if ($label_key) {
        $or_group->condition($label_key, $current_value, 'CONTAINS');
      }

      // Check if current value is numeric for ID search.
      if (is_numeric($current_value)) {
        $id_key = $this->entityTypeManager
          ->getDefinition($entity_type)
          ->getKey('id');
        $or_group->condition($id_key, (int) $current_value);
      }

      $query->condition($or_group);
    }

    // Execute query.
    $entity_ids = $query->execute();

    // Load entities and generate URLs.
    $entities = $this->entityTypeManager
      ->getStorage($entity_type)
      ->loadMultiple($entity_ids);

    $entities_with_urls = array_filter(
      $entities,
      static fn(EntityInterface $entity) => $entity->hasLinkTemplate('canonical') && $entity->access('view'),
    );
    return array_values(array_map(
      static fn (EntityInterface $entity) => $entity->toUrl('canonical', ['absolute' => TRUE])->toString(),
      $entities_with_urls,
    ));
  }

}
