<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\PromptArgumentCompletionProvider;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider;
use Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderBase;

/**
 * Provides completions from a static list of values.
 */
#[PromptArgumentCompletionProvider(
  id: 'static_list',
  label: new TranslatableMarkup('Static list'),
  description: new TranslatableMarkup('Provides completions from a static list of values configured per argument.'),
)]
final class StaticListCompletionProvider extends PromptArgumentCompletionProviderBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'values' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['values'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Completion values'),
      '#description' => $this->t('Enter one completion value per line.'),
      '#default_value' => implode("\n", $this->configuration['values'] ?? []),
      '#rows' => 5,
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('values');
    $lines = array_filter(array_map('trim', explode("\n", $values)));

    if (empty($lines)) {
      $form_state->setError(
        $form['values'],
        $this->t('At least one completion value is required.')
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('values') ?? '';
    $lines = array_filter(array_map('trim', explode("\n", $values)));
    $this->configuration['values'] = array_values($lines);
  }

  /**
   * {@inheritdoc}
   */
  public function getCompletions(string $current_value, array $configuration): array {
    $values = $this->configuration['values'] ?? [];
    if (empty($current_value)) {
      return $values;
    }
    return array_values(array_filter(
      $values,
      fn($value) => str_starts_with($value, $current_value)
    ));
  }

}
