<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Plugin\Discovery\AttributeClassDiscovery;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Manages Prompt Argument Completion Provider plugins.
 */
final class PromptArgumentCompletionProviderManager extends DefaultPluginManager {

  /**
   * Constructs a PromptArgumentCompletionProviderManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
    private readonly ContainerInterface $container,
  ) {
    parent::__construct(
      'Plugin/PromptArgumentCompletionProvider',
      $namespaces,
      $module_handler,
      'Drupal\mcp_server\Plugin\PromptArgumentCompletionProviderInterface',
      'Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider'
    );

    // Use attribute-based discovery for PHP 8 attributes.
    $this->discovery = new AttributeClassDiscovery(
      $this->subdir,
      $namespaces,
      'Drupal\mcp_server\Attribute\PromptArgumentCompletionProvider'
    );

    $this->alterInfo('mcp_server_prompt_argument_completion_provider');
    $this->setCacheBackend($cache_backend, 'mcp_server_prompt_argument_completion_provider_plugins');
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    $plugin_definition = $this->getDefinition($plugin_id);
    $plugin_class = $plugin_definition['class'];

    // Use create() method if it exists for dependency injection.
    if (is_callable([$plugin_class, 'create'])) {
      $instance = $plugin_class::create(
        $this->container,
        $configuration,
        $plugin_id,
        $plugin_definition
      );
    }
    else {
      $instance = new $plugin_class($configuration, $plugin_id, $plugin_definition);
    }

    return $instance;
  }

}
