<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin\ResourceTemplate;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\jsonapi\JsonApiResource\JsonApiDocumentTopLevel;
use Drupal\jsonapi\JsonApiResource\LinkCollection;
use Drupal\jsonapi\JsonApiResource\NullIncludedData;
use Drupal\jsonapi\JsonApiResource\ResourceObject;
use Drupal\jsonapi\JsonApiResource\ResourceObjectData;
use Drupal\jsonapi\ResourceType\ResourceTypeRepositoryInterface;
use Drupal\mcp_server\Attribute\ResourceTemplate;
use Drupal\mcp_server\Plugin\ResourceTemplateBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Provides access to content entities with canonical URLs via MCP resources.
 *
 * This plugin discovers all content entity types that have canonical link
 * templates and exposes them through the MCP resource system using the
 * drupal://entity/{entity_type}/{entity_id} URI scheme. Entity data is
 * serialized using JSON:API format and respects Drupal's entity access
 * control system.
 */
#[ResourceTemplate(
  id: 'content_entity',
  label: new TranslatableMarkup('Content Entity Resources'),
  description: new TranslatableMarkup('Provides access to content entities with canonical URLs'),
  module_dependencies: ['jsonapi'],
)]
final class ContentEntityResourceTemplate extends ResourceTemplateBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a ContentEntityResourceTemplate object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Symfony\Component\Serializer\SerializerInterface $serializer
   *   The JSON:API serializer service.
   * @param \Drupal\jsonapi\ResourceType\ResourceTypeRepositoryInterface $resourceTypeRepository
   *   The JSON:API resource type repository.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    AccountProxyInterface $currentUser,
    private readonly SerializerInterface $serializer,
    private readonly ResourceTypeRepositoryInterface $resourceTypeRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $currentUser);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('jsonapi.serializer'),
      $container->get('jsonapi.resource_type.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'require_canonical_url' => TRUE,
      'denied_entity_types' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['require_canonical_url'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require entities to have canonical URL'),
      '#default_value' => $this->configuration['require_canonical_url'] ?? TRUE,
      '#description' => $this->t('When checked, only entities with canonical link templates will be exposed as resources.'),
    ];

    // Build entity type options.
    $entity_type_options = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
        $entity_type_options[$entity_type_id] = $entity_type->getLabel();
      }
    }

    $form['denied_entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Denied entity types'),
      '#options' => $entity_type_options,
      '#default_value' => $this->configuration['denied_entity_types'] ?? [],
      '#description' => $this->t('Select entity types to exclude from MCP resource exposure.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['require_canonical_url'] = (bool) $form_state->getValue('require_canonical_url');

    // Filter out unchecked items (FALSE values) and reindex.
    $denied = array_filter($form_state->getValue('denied_entity_types', []));
    $this->configuration['denied_entity_types'] = array_values($denied);
  }

  /**
   * {@inheritdoc}
   */
  public function getUriTemplate(): string {
    return 'drupal://entity/{entity_type}/{entity_id}';
  }

  /**
   * {@inheritdoc}
   */
  public function getResources(): array {
    $resources = [];

    // Get all entity type definitions.
    $entity_types = $this->entityTypeManager->getDefinitions();

    foreach ($entity_types as $entity_type_id => $entity_type) {
      // Must be content entity with canonical link template.
      if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
        continue;
      }

      // Check canonical URL requirement if configured.
      if ($this->configuration['require_canonical_url'] ?? TRUE) {
        if (!$entity_type->hasLinkTemplate('canonical')) {
          continue;
        }
      }

      // Check deny list.
      if (in_array($entity_type_id, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
        continue;
      }

      // Generate resource metadata.
      $id_key = $entity_type->getKey('id');
      $label = $entity_type->getLabel();

      // Skip entity types without valid ID keys.
      if (empty($id_key)) {
        continue;
      }

      // Prefix resource name with plugin ID to namespace it.
      $resource_name = $this->pluginId . '__' . $entity_type_id;

      $resources[] = [
        'uri' => "drupal://entity/{$entity_type_id}/{{$id_key}}",
        'name' => $resource_name,
        'description' => 'Access ' . $label . ' entities via JSON:API',
        'mimeType' => 'application/vnd.api+json',
      ];
    }

    return $resources;
  }

  /**
   * {@inheritdoc}
   */
  public function getResourceContent(string $uri): ?array {
    $parsed = $this->parseUri($uri);
    if ($parsed === NULL) {
      return NULL;
    }

    $entity_type = $parsed['entity_type'];
    $entity_id = $parsed['entity_id'];

    // Validate entity type exists and has canonical link template.
    try {
      $definition = $this->entityTypeManager->getDefinition($entity_type);
    }
    catch (\Exception $e) {
      return NULL;
    }

    // Check deny list.
    if (in_array($entity_type, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
      return NULL;
    }

    // Check canonical URL requirement if configured.
    if ($this->configuration['require_canonical_url'] ?? TRUE) {
      if (!$definition->hasLinkTemplate('canonical')) {
        return NULL;
      }
    }

    // Load entity.
    try {
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->load($entity_id);
    }
    catch (\Exception $e) {
      return NULL;
    }

    if ($entity === NULL) {
      return NULL;
    }

    // Check access before serializing.
    if (!$entity->access('view', $this->currentUser)) {
      return NULL;
    }

    // Serialize via JSON:API.
    try {
      // Get the JSON:API resource type for this entity.
      $resource_type = $this->resourceTypeRepository->get(
        $entity->getEntityTypeId(),
        $entity->bundle()
      );

      // Create a ResourceObject from the entity.
      $resource_object = ResourceObject::createFromEntity($resource_type, $entity);

      // Wrap in a JSON:API document structure.
      $document = new JsonApiDocumentTopLevel(
        new ResourceObjectData([$resource_object], 1),
        new NullIncludedData(),
        new LinkCollection([])
      );

      // Normalize the document.
      $normalized = $this->serializer instanceof NormalizerInterface
        ? $this->serializer->normalize($document, 'api_json')
        : [];

      // JSON:API normalizer may return a CacheableNormalization object.
      // Call getNormalization() if available, otherwise use as-is.
      if (is_object($normalized) && method_exists($normalized, 'getNormalization')) {
        $normalized = $normalized->getNormalization();
      }

      return is_array($normalized) ? $normalized : NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface {
    $parsed = $this->parseUri($uri);
    if ($parsed === NULL) {
      return AccessResult::forbidden('Invalid URI format');
    }

    $entity_type = $parsed['entity_type'];
    $entity_id = $parsed['entity_id'];

    // Validate entity type exists and has canonical link template.
    try {
      $definition = $this->entityTypeManager->getDefinition($entity_type);
    }
    catch (\Exception $e) {
      return AccessResult::forbidden('Unknown entity type');
    }

    // Check deny list.
    if (in_array($entity_type, $this->configuration['denied_entity_types'] ?? [], TRUE)) {
      return AccessResult::forbidden('Entity type is denied');
    }

    // Check canonical URL requirement if configured.
    if ($this->configuration['require_canonical_url'] ?? TRUE) {
      if (!$definition->hasLinkTemplate('canonical')) {
        return AccessResult::forbidden('Entity type does not support canonical URLs');
      }
    }

    // Load entity.
    try {
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->load($entity_id);
    }
    catch (\Exception $e) {
      return AccessResult::forbidden('Failed to load entity');
    }

    if ($entity === NULL) {
      return AccessResult::forbidden('Entity not found');
    }

    // Delegate to entity access system, preserving cache metadata.
    return $entity->access('view', $account, TRUE);
  }

}
