<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Resource Template plugins.
 *
 * Provides common functionality and helper methods for resource template
 * plugins that expose Drupal data through the MCP protocol.
 */
abstract class ResourceTemplateBase extends PluginBase implements ResourceTemplateInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Constructs a ResourceTemplateBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountProxyInterface $currentUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    // @phpstan-ignore new.static
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getResourceType(): string {
    return $this->pluginDefinition['id'];
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle(): TranslatableMarkup {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): ?TranslatableMarkup {
    return $this->pluginDefinition['description'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getDependencies(): array {
    return $this->pluginDefinition['module_dependencies'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface {
    // Default implementation denies access - subclasses must override.
    return AccessResult::forbidden('Resource access not implemented');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(): bool {
    return $this->configuration['enabled'] ?? FALSE;
  }

  /**
   * Provides default configuration for this plugin.
   *
   * Subclasses can override this method to provide plugin-specific default
   * configuration values.
   *
   * @return array
   *   The default configuration array.
   */
  protected function defaultConfiguration(): array {
    return [
      'enabled' => FALSE,
    ];
  }

  /**
   * Builds configuration form elements for this plugin.
   *
   * Subclasses can override this method to provide plugin-specific
   * configuration options in the unified settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array
   *   The modified form array with plugin configuration elements.
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * Validates the configuration form.
   *
   * Subclasses can override this method to provide plugin-specific validation
   * logic for configuration form elements.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No validation by default.
  }

  /**
   * Handles submission of the configuration form.
   *
   * Subclasses can override this method to process submitted configuration
   * values before they are saved.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No additional processing by default.
  }

  /**
   * Parses a resource URI into components.
   *
   * This helper method extracts entity type and entity ID from URIs following
   * the pattern 'drupal://entity/{entity_type}/{entity_id}'.
   *
   * @param string $uri
   *   URI in format 'drupal://entity/{entity_type}/{entity_id}'.
   *
   * @return array{entity_type: string, entity_id: string}|null
   *   Associative array with 'entity_type' and 'entity_id' keys, or NULL if
   *   the URI doesn't match the expected pattern.
   */
  protected function parseUri(string $uri): ?array {
    // Pattern: drupal://entity/{entity_type}/{entity_id}.
    if (!preg_match('#^drupal://entity/([^/]+)/(.+)$#', $uri, $matches)) {
      return NULL;
    }

    return [
      'entity_type' => $matches[1],
      'entity_id' => $matches[2],
    ];
  }

}
