<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Interface for Resource Template plugins.
 *
 * Resource templates provide MCP resources through a plugin system, allowing
 * different types of Drupal data to be exposed as MCP resources.
 */
interface ResourceTemplateInterface extends PluginInspectionInterface {

  /**
   * Returns the resource template type identifier.
   *
   * @return string
   *   The resource type (e.g., 'content_entity', 'config_entity').
   */
  public function getResourceType(): string;

  /**
   * Returns the human-readable title.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title of this resource template.
   */
  public function getTitle(): TranslatableMarkup;

  /**
   * Returns the description.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   The description of this resource template, or NULL if none.
   */
  public function getDescription(): ?TranslatableMarkup;

  /**
   * Returns array of module machine names this resource depends on.
   *
   * @return array
   *   Array of module machine names (e.g., ['jsonapi', 'views']).
   */
  public function getDependencies(): array;

  /**
   * Returns the URI template pattern.
   *
   * @return string
   *   The URI template (e.g., 'drupal://entity/{entity_type}/{entity_id}').
   */
  public function getUriTemplate(): string;

  /**
   * Returns all available resources provided by this template.
   *
   * @return array
   *   Array of resource definitions with keys:
   *   - uri: string - The resource URI
   *   - name: string - Human-readable name
   *   - description: string - Resource description
   *   - mimeType: string - MIME type of resource content
   */
  public function getResources(): array;

  /**
   * Gets the content for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   *
   * @return array|null
   *   Resource content array, or NULL if not found.
   */
  public function getResourceContent(string $uri): ?array;

  /**
   * Checks access for a specific resource URI.
   *
   * @param string $uri
   *   The resource URI.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result with cacheability metadata.
   */
  public function checkAccess(string $uri, AccountInterface $account): AccessResultInterface;

  /**
   * Gets the current plugin configuration.
   *
   * @return array
   *   The plugin configuration array.
   */
  public function getConfiguration(): array;

  /**
   * Sets the plugin configuration.
   *
   * @param array $configuration
   *   The plugin configuration array.
   */
  public function setConfiguration(array $configuration): void;

  /**
   * Checks if this plugin is enabled.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isEnabled(): bool;

}
