<?php

declare(strict_types=1);

namespace Drupal\mcp_server\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Plugin\Discovery\AttributeClassDiscovery;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Manages Resource Template plugins.
 *
 * This plugin manager discovers and manages resource template plugins that
 * expose different types of Drupal data through the MCP protocol.
 */
final class ResourceTemplateManager extends DefaultPluginManager {

  /**
   * Constructs a ResourceTemplateManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container for dependency injection.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
    private readonly ContainerInterface $container,
  ) {
    parent::__construct(
      'Plugin/ResourceTemplate',
      $namespaces,
      $module_handler,
      'Drupal\mcp_server\Plugin\ResourceTemplateInterface',
      'Drupal\mcp_server\Attribute\ResourceTemplate'
    );

    // Use attribute-based discovery for PHP 8 attributes.
    $this->discovery = new AttributeClassDiscovery(
      $this->subdir,
      $namespaces,
      'Drupal\mcp_server\Attribute\ResourceTemplate'
    );

    $this->alterInfo('mcp_server_resource_template');
    $this->setCacheBackend($cache_backend, 'mcp_server_resource_template_plugins', ['mcp_server:resource_templates']);
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    $plugin_definition = $this->getDefinition($plugin_id);
    $plugin_class = $plugin_definition['class'];

    // Use create() method if it exists for dependency injection.
    if (is_callable([$plugin_class, 'create'])) {
      $instance = $plugin_class::create(
        $this->container,
        $configuration,
        $plugin_id,
        $plugin_definition
      );
    }
    else {
      $instance = new $plugin_class($configuration, $plugin_id, $plugin_definition);
    }

    return $instance;
  }

  /**
   * Gets all available resource template plugin definitions.
   *
   * @return array
   *   Array of plugin definitions keyed by plugin ID.
   */
  public function getAvailableDefinitions(): array {
    return $this->getDefinitions();
  }

}
